# 绩效计分系统

一个基于 Vue 3 + FastAPI 的现代化全栈绩效管理系统，支持用户管理、机构管理、图片上传统计和绩效评分。

## 🚀 项目概述

绩效计分系统是一个企业级全栈Web应用程序，采用前后端分离架构，提供完整的用户绩效管理、机构数据管理和图片上传统计功能。系统支持多种部署方式，具备完善的数据迁移机制和实时数据同步能力。

### ✨ 核心特性

- 🔐 **JWT认证系统** - 基于JWT的安全认证，支持token刷新
- 👥 **用户权限管理** - 管理员和普通用户角色，细粒度权限控制
- 🏢 **机构管理** - 完整的机构CRUD操作，支持批量导入和权限分配
- 📊 **实时数据统计** - 动态统计和历史数据分析，支持数据可视化
- 📈 **智能绩效评分** - 自动计算互动分数和绩效分数
- 📁 **多格式数据导出** - 支持CSV、Excel、ZIP等格式导出
- 🔄 **自动化月度重置** - 定时任务支持，数据备份和清理
- 📱 **响应式设计** - 适配桌面端和移动端
- 🐳 **容器化部署** - 支持Docker一键部署
- 🔧 **数据库迁移** - 完善的数据库版本管理和迁移机制

## 🛠️ 技术架构

### 前端技术栈
- **Vue 3** - 组合式API，响应式框架
- **Element Plus** - 企业级UI组件库
- **Pinia** - 轻量级状态管理
- **Vue Router** - 单页面路由管理
- **Vite** - 快速构建工具
- **Axios** - HTTP客户端

### 后端技术栈
- **FastAPI** - 现代Python Web框架
- **SQLAlchemy** - ORM数据库操作
- **PostgreSQL** - 关系型数据库
- **Pydantic** - 数据验证和序列化
- **JWT** - 安全认证机制
- **APScheduler** - 定时任务调度
- **Loguru** - 结构化日志

### 基础设施
- **Docker** - 容器化部署
- **Nginx** - 反向代理和静态文件服务
- **PostgreSQL** - 数据持久化
- **数据库迁移系统** - 自动化schema管理

## � 快速开始

### 环境要求

- **Node.js** 18+
- **Python** 3.11+
- **PostgreSQL** 15+
- **Docker** (可选，用于容器化部署)

### 🎯 启动方式

项目支持两种启动方式：**分步启动**和**Docker一键启动**

#### 方式一：分步启动（开发推荐）

**1. 启动数据库**
```bash
# 使用Docker启动PostgreSQL
docker run -d \
  --name performance_postgres \
  -e POSTGRES_DB=performance_db \
  -e POSTGRES_USER=performance_user \
  -e POSTGRES_PASSWORD=performance_pass \
  -p 5431:5432 \
  postgres:15-alpine
```

**2. 启动后端服务**
```bash
cd backend
pip install -r requirements.txt
# 或使用uv（推荐）
uv pip install -r requirements.txt

# 运行数据库迁移
python migrate.py

# 启动API服务
uvicorn main:app --host 0.0.0.0 --port 8000 --reload
```

**3. 启动前端服务**
```bash
npm install
npm run dev
```

#### 方式二：Docker一键启动（生产推荐）

```bash
# 克隆项目
git clone <repository-url>
cd performance-score

# 配置环境变量
cp .env.example .env
# 编辑 .env 文件，配置数据库等信息

# 一键启动所有服务
docker compose up -d

# 查看服务状态
docker compose ps
```

### 🔗 访问地址

- **前端应用**：http://localhost:4001
- **后端API**：http://localhost:8000
- **API文档**：http://localhost:8000/docs
- **数据库**：localhost:5431

### 👤 默认账号

- **管理员**：admin / admin123
- **普通用户**：user1 / user123

## 📁 项目结构

```
performance-score/
├── frontend/              # 前端应用
│   ├── src/              # 源代码目录
│   │   ├── components/   # Vue组件
│   │   ├── views/        # 页面组件
│   │   │   ├── admin/    # 管理员页面
│   │   │   ├── auth/     # 认证页面
│   │   │   └── user/     # 用户页面
│   │   ├── store/        # Pinia状态管理
│   │   │   ├── auth.js   # 认证状态
│   │   │   └── data.js   # 数据状态
│   │   ├── router/       # 路由配置
│   │   ├── services/     # API服务
│   │   │   └── api.js    # API接口封装
│   │   ├── utils/        # 工具函数
│   │   └── styles/       # 样式文件
│   ├── package.json      # 前端依赖配置
│   └── vite.config.js    # Vite构建配置
├── backend/              # 后端API服务
│   ├── routers/          # API路由
│   │   ├── auth.py       # 认证路由
│   │   ├── users.py      # 用户管理
│   │   ├── institutions.py # 机构管理
│   │   ├── history.py    # 历史数据
│   │   └── system_config.py # 系统配置
│   ├── models/           # 数据模型
│   ├── migrations/       # 数据库迁移
│   │   ├── migrate.py    # 迁移脚本
│   │   └── versions/     # 迁移版本
│   ├── database.py       # 数据库配置
│   ├── main.py          # FastAPI应用入口
│   └── requirements.txt  # Python依赖
├── docker compose.yml    # Docker编排配置
├── Dockerfile           # 容器构建文件
├── .env.example         # 环境变量模板
└── README.md           # 项目说明
```

## 🔧 数据库迁移策略

项目采用完善的数据库版本管理和迁移机制，确保数据库结构变更的安全性和一致性。

### 迁移系统特点

- **🔄 自动迁移检测** - 启动时自动检测并执行待迁移的版本
- **📝 版本化管理** - 每个数据库变更都有对应的迁移文件
- **🛡️ 安全回滚** - 支持迁移失败时的安全回滚机制
- **📊 迁移日志** - 详细记录每次迁移的执行状态和时间

### 迁移文件结构

```
backend/migrations/
├── migrate.py           # 迁移执行器
├── versions/           # 迁移版本目录
│   ├── 001_initial.py  # 初始化数据库结构
│   ├── 002_add_institutions_data.py # 添加机构数据字段
│   └── 003_add_indexes.py # 添加数据库索引
└── migration_log.txt   # 迁移执行日志
```

### 使用方法

```bash
# 手动执行迁移
cd backend
python migrate.py

# 查看迁移状态
python migrate.py --status

# 回滚到指定版本
python migrate.py --rollback 001
```

### 新增迁移

当需要修改数据库结构时：

1. 在 `backend/migrations/versions/` 目录下创建新的迁移文件
2. 按照版本号命名：`00X_description.py`
3. 实现 `upgrade()` 和 `downgrade()` 函数
4. 重启应用，迁移将自动执行

## 🎯 核心功能模块

### 🔐 认证与权限系统
- **JWT安全认证** - 基于Token的无状态认证
- **角色权限控制** - 管理员/普通用户权限分离
- **Token自动刷新** - 无感知的登录状态保持
- **安全登出** - 服务端Token失效机制

### 👥 用户管理系统
- **用户CRUD操作** - 完整的用户生命周期管理
- **批量用户导入** - 支持Excel/CSV批量导入
- **用户状态管理** - 启用/禁用用户账号
- **密码安全策略** - 密码加密存储和重置

### 🏢 机构管理系统
- **机构层级管理** - 支持多级机构结构
- **权限精细控制** - 机构级别的数据访问控制
- **批量机构操作** - 导入、导出、转移等批量操作
- **机构数据统计** - 实时统计机构相关数据

### 📊 数据统计分析
- **实时数据监控** - 用户活跃度、上传量等实时统计
- **历史数据分析** - 月度、季度、年度数据趋势
- **绩效评分算法** - 智能计算用户绩效分数
- **数据可视化** - 图表展示统计结果

### 📁 数据导出系统
- **多格式导出** - CSV、Excel、ZIP等格式支持
- **自定义导出** - 可选择导出字段和时间范围
- **批量文件打包** - 图片和数据文件批量下载
- **导出任务管理** - 大数据量异步导出

### 🔄 自动化任务系统
- **月度数据重置** - 定时清理和备份月度数据
- **数据备份策略** - 自动备份重要数据
- **系统维护任务** - 日志清理、缓存更新等
- **任务调度管理** - 灵活的定时任务配置

## 📚 API文档

### 🔗 前后端接口标准约定

#### 📋 统一响应格式

**标准响应格式（所有接口）：**
```json
{
  "success": true,
  "message": "操作成功",
  "data": [实际数据内容]
}
```

**特殊响应格式（认证相关接口）：**
```json
{
  "success": true,
  "access_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
  "refresh_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
  "token_type": "bearer",
  "expires_in": 86400,
  "user": {...},
  "message": "登录成功"
}
```

#### 🔧 前端自动解包机制

前端API客户端自动处理响应格式：
- **自动检测**：识别 `BaseResponse` 格式（包含 `success` 字段）
- **自动解包**：提取 `data` 字段内容，透明返回给业务代码
- **错误处理**：检查 `success` 字段，失败时自动抛出异常
- **兼容性**：对特殊格式接口（如login）保持原样返回

#### 📊 状态码约定

| 状态码 | 含义 | 使用场景 |
|--------|------|----------|
| 200 | 成功 | 正常请求成功 |
| 400 | 请求错误 | 参数验证失败、业务逻辑错误 |
| 401 | 未授权 | Token无效、未登录 |
| 403 | 禁止访问 | 权限不足 |
| 404 | 资源不存在 | 数据不存在 |
| 500 | 服务器错误 | 系统内部错误 |

#### 🔐 认证机制

- **认证方式**：JWT Bearer Token
- **Token类型**：Access Token（短期）+ Refresh Token（长期）
- **自动刷新**：前端自动检测401错误并尝试刷新Token
- **安全登出**：服务端Token黑名单机制

### 接口概览

系统提供完整的RESTful API，支持前后端分离架构。

**基础信息**
- **API Base URL**: `http://localhost:8000/api`
- **认证方式**: Bearer Token (JWT)
- **数据格式**: JSON
- **API文档**: http://localhost:8000/docs (Swagger UI)

### 主要接口

| 模块 | 方法 | 路径 | 描述 | 响应格式 |
|------|------|------|------|----------|
| 认证 | POST | `/users/login` | 用户登录 | 特殊格式 |
| 认证 | POST | `/users/logout` | 用户登出 | 标准格式 |
| 认证 | POST | `/users/refresh` | 刷新Token | 特殊格式 |
| 用户 | GET | `/users` | 获取用户列表 | 标准格式 |
| 用户 | GET | `/users/{id}` | 获取用户信息 | 标准格式 |
| 用户 | POST | `/users` | 创建用户 | 标准格式 |
| 用户 | PUT | `/users/{id}` | 更新用户 | 标准格式 |
| 用户 | DELETE | `/users/{id}` | 删除用户 | 标准格式 |
| 机构 | GET | `/institutions` | 获取机构列表 | 标准格式 |
| 机构 | GET | `/institutions/{id}` | 获取机构信息 | 标准格式 |
| 机构 | POST | `/institutions` | 创建机构 | 标准格式 |
| 机构 | PUT | `/institutions/{id}` | 更新机构 | 标准格式 |
| 机构 | DELETE | `/institutions/{id}` | 删除机构 | 标准格式 |
| 历史 | GET | `/history` | 获取历史数据列表 | 标准格式 |
| 历史 | GET | `/history/{month}` | 获取指定月份历史 | 标准格式 |
| 历史 | POST | `/history` | 保存月度历史 | 标准格式 |
| 历史 | DELETE | `/history` | 清空历史数据 | 标准格式 |
| 配置 | GET | `/config` | 获取系统配置 | 标准格式 |
| 配置 | GET | `/config/{key}` | 获取指定配置项 | 标准格式 |

### 🔧 API使用示例

#### 认证流程

```bash
# 1. 用户登录（特殊格式响应）
curl -X POST "http://localhost:8000/api/users/login" \
  -H "Content-Type: application/json" \
  -d '{"phone":"admin","password":"admin123"}'

# 响应示例：
{
  "success": true,
  "access_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
  "refresh_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
  "token_type": "bearer",
  "expires_in": 86400,
  "user": {
    "id": "admin",
    "phone": "admin",
    "name": "系统管理员",
    "role": "admin"
  },
  "message": "登录成功"
}

# 2. 使用Token访问API（标准格式响应）
curl -X GET "http://localhost:8000/api/users" \
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN"

# 响应示例：
{
  "success": true,
  "message": "获取用户列表成功",
  "data": [
    {
      "id": "admin",
      "phone": "admin",
      "name": "系统管理员",
      "role": "admin",
      "institutions": [],
      "created_at": "2025-08-30T10:00:00",
      "updated_at": "2025-08-30T10:00:00"
    }
  ]
}

# 3. 刷新Token
curl -X POST "http://localhost:8000/api/users/refresh" \
  -H "Content-Type: application/json" \
  -d '{"refresh_token":"YOUR_REFRESH_TOKEN"}'

# 4. 用户登出
curl -X POST "http://localhost:8000/api/users/logout" \
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN"
```

#### 业务接口示例

```bash
# 获取机构列表
curl -X GET "http://localhost:8000/api/institutions" \
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN"

# 创建机构
curl -X POST "http://localhost:8000/api/institutions" \
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "id": "inst_001",
    "name": "测试机构",
    "institution_id": "TEST001",
    "owner_id": "user_001"
  }'

# 获取历史数据
curl -X GET "http://localhost:8000/api/history" \
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN"

# 获取系统配置
curl -X GET "http://localhost:8000/api/config" \
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN"
```

## 🔧 环境配置

### 环境变量

创建 `.env` 文件配置环境变量：

```bash
# 数据库配置
DATABASE_URL=postgresql://performance_user:performance_pass@localhost:5431/performance_db

# JWT配置
SECRET_KEY=your-secret-key-here
ACCESS_TOKEN_EXPIRE_MINUTES=30
REFRESH_TOKEN_EXPIRE_DAYS=7

# 应用配置
DEBUG=true
CORS_ORIGINS=["http://localhost:4001"]
```

### 开发环境配置

前端开发服务器配置（`vite.config.js`）：

```javascript
export default defineConfig({
  server: {
    port: 4001,
    host: '0.0.0.0',
    proxy: {
      '/api': {
        target: 'http://localhost:8000',
        changeOrigin: true
      }
    }
  }
})
```

## 🚀 生产部署

### Docker Compose 部署（推荐）

```bash
# 生产环境一键部署
docker compose -f docker compose.prod.yml up -d

# 查看服务状态
docker compose ps

# 查看日志
docker compose logs -f
```

### 手动部署

**后端部署**
```bash
cd backend
pip install -r requirements.txt
python migrate.py
gunicorn main:app -w 4 -k uvicorn.workers.UvicornWorker --bind 0.0.0.0:8000
```

**前端部署**
```bash
npm run build
# 将 dist/ 目录部署到 Nginx 或其他静态文件服务器
```

### Nginx 配置示例

```nginx
server {
    listen 80;
    server_name your-domain.com;

    # 前端静态文件
    location / {
        root /path/to/dist;
        try_files $uri $uri/ /index.html;
    }

    # 后端API代理
    location /api/ {
        proxy_pass http://localhost:8000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
    }
}

## 📝 使用指南

### 🔑 首次使用

1. **启动系统** - 选择合适的启动方式（分步启动或Docker）
2. **访问应用** - 打开浏览器访问 http://localhost:4001
3. **管理员登录** - 使用默认管理员账号：admin / admin123
4. **初始化数据** - 创建用户和机构，配置系统参数

### 👨‍💼 管理员操作流程

**用户管理**
- 创建用户账号，分配角色权限
- 批量导入用户数据（支持Excel/CSV）
- 管理用户状态（启用/禁用）
- 重置用户密码

**机构管理**
- 创建机构层级结构
- 分配机构给用户
- 批量导入机构数据
- 设置机构权限

**数据管理**
- 查看实时统计数据
- 导出各类报表
- 执行月度数据重置
- 管理历史数据

### 👤 普通用户操作流程

**日常操作**
- 登录个人账号
- 查看分配的机构
- 上传机构相关图片
- 查看个人绩效统计

**数据查看**
- 个人上传统计
- 绩效分数排名
- 历史数据查询

## ⚠️ 注意事项

### 🔒 安全相关
- **生产环境密钥** - 务必修改默认的JWT密钥和数据库密码
- **HTTPS部署** - 生产环境建议使用HTTPS协议
- **权限控制** - 严格控制管理员权限的分配
- **定期备份** - 建立完善的数据备份策略

### 🛠️ 技术要求
- **浏览器兼容** - 推荐使用Chrome 90+、Firefox 88+、Safari 14+
- **数据库维护** - 定期清理日志和优化数据库性能
- **资源监控** - 监控服务器资源使用情况
- **版本管理** - 使用数据库迁移系统管理结构变更

### 📊 运维建议
- **监控告警** - 设置系统监控和告警机制
- **日志管理** - 定期清理和归档系统日志
- **性能优化** - 根据使用情况调整数据库和应用配置
- **容灾备份** - 建立异地备份和容灾恢复方案

## 🔄 版本更新

### v2.0.0 (2025-08-30) - 架构重构版本

#### 🎯 重大变更
- **全栈重构** - 从单体前端应用重构为前后端分离架构
- **数据库升级** - 从LocalStorage迁移到PostgreSQL数据库
- **认证系统** - 实现基于JWT的安全认证机制
- **API设计** - 设计完整的RESTful API接口

#### ✨ 新增功能
- **Docker支持** - 支持容器化部署和一键启动
- **数据库迁移** - 完善的数据库版本管理和迁移机制
- **API文档** - 自动生成的Swagger API文档
- **日志系统** - 结构化日志记录和错误追踪
- **定时任务** - 支持自动化的后台任务调度

#### 🔧 技术升级
- **后端框架** - 采用FastAPI构建高性能API服务
- **数据库ORM** - 使用SQLAlchemy进行数据库操作
- **状态管理** - 前端使用Pinia替代Vuex
- **构建工具** - 使用Vite提升开发和构建效率

#### 🚀 部署优化
- **多种部署方式** - 支持分步启动和Docker一键部署
- **环境配置** - 完善的环境变量配置管理
- **生产优化** - 针对生产环境的性能和安全优化

## �️ 故障排除

### 常见问题

**启动问题**
```bash
# 检查端口占用
lsof -i :8000  # 后端端口
lsof -i :4001  # 前端端口
lsof -i :5431  # 数据库端口

# 清理Docker容器
docker compose down
docker system prune -f
```

**数据库连接问题**
```bash
# 检查数据库状态
docker ps | grep postgres

# 查看数据库日志
docker logs performance_postgres

# 重新创建数据库
docker compose down
docker compose up -d postgres
```

**API请求失败**
- 检查后端服务是否正常启动
- 确认API地址和端口配置正确
- 查看浏览器网络面板的错误信息
- 检查JWT Token是否有效

### 日志查看

```bash
# 查看后端日志
docker compose logs -f backend

# 查看前端构建日志
npm run dev

# 查看数据库日志
docker compose logs postgres
```

## �📞 技术支持

### 🔍 问题诊断步骤

1. **查看系统状态** - 确认所有服务正常运行
2. **检查日志输出** - 查看详细的错误信息
3. **验证配置** - 确认环境变量和配置文件正确
4. **测试API** - 使用Swagger文档测试API接口
5. **浏览器调试** - 检查浏览器控制台和网络面板

### 📚 参考资源

- **API文档**: http://localhost:8000/docs
- **FastAPI官方文档**: https://fastapi.tiangolo.com/
- **Vue 3官方文档**: https://vuejs.org/
- **Element Plus文档**: https://element-plus.org/

## 🤝 贡献指南

### 开发流程

1. **Fork项目** - 创建项目分支
2. **本地开发** - 在本地环境进行开发和测试
3. **代码规范** - 遵循项目的代码规范和最佳实践
4. **测试验证** - 确保新功能正常工作且不影响现有功能
5. **提交PR** - 提交Pull Request并描述变更内容

### 代码规范

- **Python**: 遵循PEP 8规范
- **JavaScript**: 使用ESLint和Prettier
- **Vue**: 遵循Vue官方风格指南
- **Git**: 使用语义化的提交信息

## 📄 许可证

本项目采用 MIT 许可证。详见 [LICENSE](LICENSE) 文件。

---

**绩效计分系统** - 现代化的企业级绩效管理解决方案

🌟 如果这个项目对您有帮助，请给我们一个Star！
