"""
数据库连接和操作层
使用 databases 和 SQLAlchemy 进行异步数据库操作
"""

import databases
import sqlalchemy
from sqlalchemy import create_engine, MetaData, Table, Column, String, Integer, Text, TIMESTAMP, Boolean, ForeignKey, LargeBinary, Numeric
from sqlalchemy.dialects.postgresql import JSONB
from sqlalchemy.sql import func
from config import settings
from loguru import logger

# 创建数据库连接
database = databases.Database(settings.DATABASE_URL)

# 创建 SQLAlchemy 引擎
engine = create_engine(settings.DATABASE_URL)

# 创建元数据对象
metadata = MetaData()

# 定义用户表
users_table = Table(
    "users",
    metadata,
    Column("id", String(50), primary_key=True),
    Column("phone", String(20), unique=True, nullable=False),
    Column("password", String(255), nullable=False),
    Column("name", String(100), nullable=False),
    Column("role", String(20), default="user"),
    Column("institutions", JSONB, default="[]"),
    Column("created_at", TIMESTAMP, server_default=func.now()),
    Column("updated_at", TIMESTAMP, server_default=func.now(), onupdate=func.now()),
)

# 定义机构表
institutions_table = Table(
    "institutions",
    metadata,
    Column("id", String(50), primary_key=True),
    Column("institution_id", String(50), unique=True),
    Column("name", String(200), nullable=False),
    Column("owner_id", String(50), ForeignKey("users.id", ondelete="SET NULL")),
    Column("created_at", TIMESTAMP, server_default=func.now()),
    Column("updated_at", TIMESTAMP, server_default=func.now(), onupdate=func.now()),
)

# 定义机构图片表
institution_images_table = Table(
    "institution_images",
    metadata,
    Column("id", String(50), primary_key=True),
    Column("institution_id", String(50), ForeignKey("institutions.id", ondelete="CASCADE"), nullable=False),
    Column("url", Text, nullable=False),
    Column("upload_time", TIMESTAMP, nullable=False),
    Column("created_at", TIMESTAMP, server_default=func.now()),
)

# 定义机构图片二进制表
institution_images_binary_table = Table(
    "institution_images_binary",
    metadata,
    Column("id", String(50), primary_key=True),
    Column("institution_id", String(50), ForeignKey("institutions.id", ondelete="CASCADE"), nullable=False),
    Column("image_data", LargeBinary, nullable=False),
    Column("original_filename", String(255)),
    Column("mime_type", String(50), nullable=False, server_default="image/jpeg"),
    Column("file_size", Integer, nullable=False),
    Column("width", Integer),
    Column("height", Integer),
    Column("compressed_quality", Numeric(3, 2), server_default="0.8"),
    Column("checksum", String(64)),
    Column("upload_time", TIMESTAMP, nullable=False, server_default=func.now()),
    Column("created_at", TIMESTAMP, server_default=func.now()),
)

# 定义系统配置表
system_config_table = Table(
    "system_config",
    metadata,
    Column("id", Integer, primary_key=True, autoincrement=True),
    Column("config_key", String(100), unique=True, nullable=False),
    Column("config_value", JSONB),
    Column("description", Text),
    Column("created_at", TIMESTAMP, server_default=func.now()),
    Column("updated_at", TIMESTAMP, server_default=func.now(), onupdate=func.now()),
)

# 定义月度历史统计表
monthly_history_table = Table(
    "monthly_history",
    metadata,
    Column("id", Integer, primary_key=True, autoincrement=True),
    Column("month", String(7), unique=True, nullable=False),
    Column("save_time", TIMESTAMP(timezone=True), nullable=False),
    Column("total_users", Integer, nullable=False),
    Column("total_institutions", Integer, nullable=False),
    Column("total_images", Integer, nullable=False),
    Column("user_stats", JSONB, nullable=False),
    Column("institutions_data", JSONB, nullable=True),  # 新增：存储完整的机构和图片数据
    Column("created_at", TIMESTAMP(timezone=True), server_default=func.now()),
)


class DatabaseManager:
    """数据库管理器，提供通用的数据库操作方法"""
    
    def __init__(self):
        self.database = database
    
    async def fetch_one(self, query, values=None):
        """执行查询并返回单条记录"""
        try:
            return await self.database.fetch_one(query, values)
        except Exception as e:
            logger.error(f"数据库查询失败: {e}")
            raise
    
    async def fetch_all(self, query, values=None):
        """执行查询并返回所有记录"""
        try:
            return await self.database.fetch_all(query, values)
        except Exception as e:
            logger.error(f"数据库查询失败: {e}")
            raise
    
    async def execute(self, query, values=None):
        """执行 INSERT/UPDATE/DELETE 操作"""
        try:
            return await self.database.execute(query, values)
        except Exception as e:
            logger.error(f"数据库操作失败: {e}")
            raise
    
    async def execute_many(self, query, values):
        """批量执行操作"""
        try:
            return await self.database.execute_many(query, values)
        except Exception as e:
            logger.error(f"批量数据库操作失败: {e}")
            raise
    
    def transaction(self):
        """创建数据库事务"""
        return self.database.transaction()


# 创建全局数据库管理器实例
db_manager = DatabaseManager()


async def get_database():
    """依赖注入：获取数据库连接"""
    return db_manager


async def check_database_connection():
    """检查数据库连接状态"""
    try:
        await database.fetch_one("SELECT 1")
        return True
    except Exception as e:
        logger.error(f"数据库连接检查失败: {e}")
        return False


async def initialize_database():
    """初始化数据库表结构"""
    try:
        # 创建所有表
        metadata.create_all(engine)
        logger.info("✅ 数据库表结构初始化完成")
        
        # 检查并插入默认数据
        await insert_default_data()
        
    except Exception as e:
        logger.error(f"数据库初始化失败: {e}")
        raise


async def insert_default_data():
    """插入默认数据"""
    try:
        # 检查是否已有管理员用户
        admin_exists = await database.fetch_one(
            users_table.select().where(users_table.c.id == "admin")
        )

        if not admin_exists:
            # 插入默认管理员用户
            await database.execute(
                users_table.insert().values(
                    id="admin",
                    phone="admin",
                    password="admin123",
                    name="系统管理员",
                    role="admin",
                    institutions=[]
                )
            )
            logger.info("✅ 默认管理员用户创建成功")
        else:
            # 检查管理员用户密码是否为空，如果为空则设置默认密码
            if not admin_exists["password"]:
                await database.execute(
                    users_table.update().where(users_table.c.id == "admin").values(
                        password="admin123"
                    )
                )
                logger.info("✅ 管理员用户密码已修复")
        
        # 检查并插入默认系统配置
        config_items = [
            ("initialized", True, "系统是否已初始化"),
            ("version", "8.8.0", "系统版本"),
            ("hasDefaultData", False, "是否有默认示例数据")
        ]
        
        for key, value, desc in config_items:
            exists = await database.fetch_one(
                system_config_table.select().where(system_config_table.c.config_key == key)
            )
            
            if not exists:
                await database.execute(
                    system_config_table.insert().values(
                        config_key=key,
                        config_value=value,
                        description=desc
                    )
                )
        
        logger.info("✅ 默认系统配置创建成功")
        
    except Exception as e:
        logger.error(f"插入默认数据失败: {e}")
        raise
