"""
绩效计分系统 FastAPI 后端服务
主要功能：
1. 提供完整的 CRUD API 接口
2. 替换前端的 localStorage 存储
3. 实现数据持久化到 PostgreSQL
4. 支持数据迁移和同步
"""

from fastapi import FastAPI, HTTPException, Depends
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from contextlib import asynccontextmanager
import os
from loguru import logger

from database import database, engine, metadata
from config import settings
from routers import users, institutions, system_config, history, migration, scheduler_config
from scheduler import monthly_scheduler


@asynccontextmanager
async def lifespan(app: FastAPI):
    """应用生命周期管理"""
    # 启动时执行
    logger.info("🚀 启动绩效计分系统 API 服务")
    
    # 连接数据库
    await database.connect()
    logger.info("✅ 数据库连接成功")
    
    # 创建表结构（如果不存在）
    metadata.create_all(engine)
    logger.info("✅ 数据库表结构检查完成")

    # 执行数据库迁移
    try:
        from migrations.loader import load_migrations
        from migrations.manager import migration_manager

        # 加载迁移文件
        migration_count = load_migrations()
        logger.info(f"✅ 加载了 {migration_count} 个迁移文件")

        # 执行迁移
        migration_result = await migration_manager.migrate_to_latest()

        if migration_result["success"]:
            if migration_result["executed_migrations"] > 0:
                logger.info(f"🎉 数据库迁移完成！执行了 {migration_result['executed_migrations']} 个迁移")
            else:
                logger.info("✅ 数据库已是最新版本，无需迁移")
        else:
            logger.error("❌ 数据库迁移失败")
            logger.error(f"失败的迁移: {migration_result.get('failed_migrations', [])}")
            # 在生产环境中，可以选择是否继续启动应用
            # 这里我们记录错误但继续启动，让管理员可以通过API手动处理

    except Exception as e:
        logger.error(f"数据库迁移过程异常: {e}")
        logger.error("应用将继续启动，但建议检查迁移状态")

    # 插入默认数据
    from database import insert_default_data
    await insert_default_data()
    logger.info("✅ 默认数据初始化完成")

    # 启动定时任务调度器
    await monthly_scheduler.start()

    yield

    # 关闭时执行
    logger.info("🔄 正在关闭 API 服务")

    # 停止定时任务调度器
    await monthly_scheduler.stop()

    await database.disconnect()
    logger.info("✅ 数据库连接已关闭")


# 创建 FastAPI 应用实例
app = FastAPI(
    title="绩效计分系统 API",
    description="为绩效计分系统提供数据持久化和 API 服务",
    version="1.0.0",
    lifespan=lifespan,
    docs_url="/docs",
    redoc_url="/redoc"
)

# 配置 CORS 中间件
# 处理 CORS_ORIGINS 配置，支持字符串、JSON数组和列表格式
cors_origins = settings.CORS_ORIGINS
allow_credentials = True

if isinstance(cors_origins, str):
    if cors_origins == "*":
        # 当允许所有来源时，不能同时设置 allow_credentials=True
        cors_origins = ["*"]
        allow_credentials = False
    elif cors_origins.startswith("[") and cors_origins.endswith("]"):
        # JSON数组格式，解析为列表
        import json
        try:
            cors_origins = json.loads(cors_origins)
        except json.JSONDecodeError:
            logger.error(f"CORS_ORIGINS JSON格式错误: {cors_origins}")
            # 回退到逗号分隔格式
            cors_origins = [origin.strip() for origin in cors_origins.split(",")]
    else:
        # 支持逗号分隔的多个域名
        cors_origins = [origin.strip() for origin in cors_origins.split(",")]

logger.info(f"🌐 CORS配置: origins={cors_origins}, credentials={allow_credentials}")

app.add_middleware(
    CORSMiddleware,
    allow_origins=cors_origins,
    allow_credentials=allow_credentials,
    allow_methods=["GET", "POST", "PUT", "DELETE", "OPTIONS"],
    allow_headers=["*"],
)

# 注册路由
app.include_router(users.router, prefix="/api/users", tags=["用户管理"])
app.include_router(institutions.router, prefix="/api/institutions", tags=["机构管理"])
app.include_router(system_config.router, prefix="/api/config", tags=["系统配置"])
app.include_router(history.router, prefix="/api/history", tags=["历史数据"])
app.include_router(migration.router, prefix="/api/migration", tags=["数据迁移"])
app.include_router(scheduler_config.router, prefix="/api/scheduler", tags=["调度器管理"])

# 导入并注册图片服务路由
from routers import images
app.include_router(images.router)


@app.get("/", summary="根路径")
async def root():
    """API 根路径，返回服务信息"""
    return {
        "message": "绩效计分系统 API 服务",
        "version": "1.0.0",
        "status": "running",
        "docs": "/docs"
    }


@app.get("/health", summary="健康检查")
async def health_check():
    """健康检查端点，用于容器健康检查"""
    try:
        # 检查数据库连接
        await database.fetch_one("SELECT 1")
        from datetime import datetime
        return {
            "status": "healthy",
            "database": "connected",
            "timestamp": datetime.now().isoformat()
        }
    except Exception as e:
        logger.error(f"健康检查失败: {e}")
        raise HTTPException(status_code=503, detail="Service unavailable")


@app.exception_handler(Exception)
async def global_exception_handler(request, exc):
    """全局异常处理器"""
    logger.error(f"未处理的异常: {exc}")
    return JSONResponse(
        status_code=500,
        content={"detail": "Internal server error", "error": str(exc)}
    )


if __name__ == "__main__":
    import uvicorn
    
    # 配置日志
    logger.add(
        "logs/api.log",
        rotation="1 day",
        retention="30 days",
        level="INFO",
        format="{time:YYYY-MM-DD HH:mm:ss} | {level} | {message}"
    )
    
    # 启动服务
    uvicorn.run(
        "main:app",
        host=settings.API_HOST,
        port=settings.API_PORT,
        reload=True if os.getenv("NODE_ENV") != "production" else False,
        log_level="info"
    )
