#!/usr/bin/env python3
"""
独立的数据库迁移工具脚本
可以在容器外或CI/CD中使用，不依赖FastAPI应用

使用方法:
    python migrate.py                    # 执行所有待执行的迁移
    python migrate.py --status           # 查看迁移状态
    python migrate.py --version          # 查看当前版本
    python migrate.py --reload           # 重新加载迁移文件
    python migrate.py --help             # 显示帮助信息

环境变量:
    DATABASE_URL: 数据库连接URL
    MIGRATION_LOG_LEVEL: 日志级别 (DEBUG, INFO, WARNING, ERROR)
"""
import asyncio
import sys
import argparse
import os
from pathlib import Path
from loguru import logger

# 添加项目根目录到Python路径
sys.path.insert(0, str(Path(__file__).parent))

def setup_logging(level: str = "INFO"):
    """设置日志配置"""
    logger.remove()  # 移除默认处理器
    
    # 添加控制台输出
    logger.add(
        sys.stdout,
        level=level,
        format="<green>{time:YYYY-MM-DD HH:mm:ss}</green> | <level>{level: <8}</level> | <cyan>{name}</cyan>:<cyan>{function}</cyan>:<cyan>{line}</cyan> - <level>{message}</level>",
        colorize=True
    )
    
    # 添加文件输出
    logger.add(
        "logs/migration.log",
        level=level,
        format="{time:YYYY-MM-DD HH:mm:ss} | {level: <8} | {name}:{function}:{line} - {message}",
        rotation="10 MB",
        retention="30 days",
        compression="zip"
    )

async def show_migration_status():
    """显示迁移状态"""
    try:
        from migrations.manager import migration_manager
        from migrations.loader import load_migrations
        
        # 加载迁移
        migration_count = load_migrations()
        logger.info(f"加载了 {migration_count} 个迁移文件")
        
        # 获取状态
        status = await migration_manager.get_migration_status()
        
        print("\n" + "="*60)
        print("📊 数据库迁移状态")
        print("="*60)
        print(f"总迁移数量: {status['total_migrations']}")
        print(f"已执行数量: {status['executed_count']}")
        print(f"待执行数量: {status['pending_count']}")
        print(f"是否最新版本: {'✅ 是' if status['is_up_to_date'] else '❌ 否'}")
        
        if status['executed_migrations']:
            print(f"\n已执行的迁移:")
            for version in status['executed_migrations']:
                print(f"  ✅ {version}")
        
        if status['pending_migrations']:
            print(f"\n待执行的迁移:")
            for migration in status['pending_migrations']:
                print(f"  ⏳ {migration['version']}: {migration['description']}")
        
        print("="*60)
        
    except Exception as e:
        logger.error(f"获取迁移状态失败: {e}")
        sys.exit(1)

async def show_current_version():
    """显示当前版本"""
    try:
        from migrations.manager import migration_manager
        
        executed_migrations = await migration_manager.get_executed_migrations()
        current_version = executed_migrations[-1] if executed_migrations else "0.0.0"
        
        print(f"\n当前数据库schema版本: {current_version}")
        print(f"已执行迁移数量: {len(executed_migrations)}")
        
    except Exception as e:
        logger.error(f"获取当前版本失败: {e}")
        sys.exit(1)

async def execute_migrations():
    """执行迁移"""
    try:
        from migrations.loader import load_migrations
        from migrations.manager import migration_manager
        
        logger.info("🚀 开始独立迁移过程")
        
        # 加载迁移
        migration_count = load_migrations()
        logger.info(f"✅ 加载了 {migration_count} 个迁移")
        
        # 执行迁移
        result = await migration_manager.migrate_to_latest()
        
        if result["success"]:
            if result["executed_migrations"] > 0:
                logger.info("🎉 迁移完成！")
                print(f"\n✅ 成功执行 {result['executed_migrations']} 个迁移")
                print(f"📋 总待执行迁移: {result.get('total_pending', 0)}")
            else:
                logger.info("✅ 数据库已是最新版本")
                print("\n✅ 数据库已是最新版本，无需迁移")
        else:
            logger.error("❌ 迁移失败！")
            print(f"\n❌ 失败的迁移: {result.get('failed_migrations', [])}")
            print(f"❌ 错误信息: {result.get('message', '未知错误')}")
            sys.exit(1)
        
    except Exception as e:
        logger.error(f"迁移过程异常: {e}")
        print(f"\n❌ 迁移异常: {e}")
        sys.exit(1)

async def reload_migrations():
    """重新加载迁移文件"""
    try:
        from migrations.loader import reload_migrations
        
        logger.info("🔄 重新加载迁移文件")
        migration_count = reload_migrations()
        
        print(f"\n✅ 成功重新加载 {migration_count} 个迁移文件")
        
    except Exception as e:
        logger.error(f"重新加载迁移文件失败: {e}")
        sys.exit(1)

async def main():
    """主函数"""
    parser = argparse.ArgumentParser(
        description="数据库迁移工具",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
示例:
    python migrate.py                    # 执行所有待执行的迁移
    python migrate.py --status           # 查看迁移状态
    python migrate.py --version          # 查看当前版本
    python migrate.py --reload           # 重新加载迁移文件
    
环境变量:
    DATABASE_URL: 数据库连接URL
    MIGRATION_LOG_LEVEL: 日志级别 (DEBUG, INFO, WARNING, ERROR)
        """
    )
    
    parser.add_argument(
        "--status", 
        action="store_true", 
        help="显示迁移状态"
    )
    
    parser.add_argument(
        "--version", 
        action="store_true", 
        help="显示当前数据库schema版本"
    )
    
    parser.add_argument(
        "--reload", 
        action="store_true", 
        help="重新加载迁移文件"
    )
    
    parser.add_argument(
        "--log-level", 
        default=os.getenv("MIGRATION_LOG_LEVEL", "INFO"),
        choices=["DEBUG", "INFO", "WARNING", "ERROR"],
        help="日志级别"
    )
    
    args = parser.parse_args()
    
    # 设置日志
    setup_logging(args.log_level)
    
    try:
        # 连接数据库
        from database import database
        await database.connect()
        logger.info("✅ 数据库连接成功")
        
        # 根据参数执行相应操作
        if args.status:
            await show_migration_status()
        elif args.version:
            await show_current_version()
        elif args.reload:
            await reload_migrations()
        else:
            # 默认执行迁移
            await execute_migrations()
        
    except Exception as e:
        logger.error(f"程序执行异常: {e}")
        print(f"\n❌ 程序异常: {e}")
        sys.exit(1)
        
    finally:
        try:
            from database import database
            await database.disconnect()
            logger.info("✅ 数据库连接已关闭")
        except:
            pass

if __name__ == "__main__":
    # 确保logs目录存在
    Path("logs").mkdir(exist_ok=True)
    
    # 运行主函数
    asyncio.run(main())
