"""
迁移加载器 - 自动发现和注册迁移
负责扫描migrations/versions目录，自动加载和注册所有迁移文件
"""
import os
import sys
import importlib
import inspect
from pathlib import Path
from typing import List, Type
from loguru import logger

from .base import Migration
from .manager import migration_manager

class MigrationLoader:
    """迁移加载器"""
    
    def __init__(self, versions_dir: str = None):
        """
        初始化迁移加载器
        
        Args:
            versions_dir: 迁移版本目录路径，默认为当前目录下的versions
        """
        if versions_dir is None:
            self.versions_dir = Path(__file__).parent / "versions"
        else:
            self.versions_dir = Path(versions_dir)
        
        self.loaded_migrations: List[Migration] = []
    
    def discover_migration_files(self) -> List[str]:
        """发现所有迁移文件"""
        if not self.versions_dir.exists():
            logger.warning(f"迁移目录不存在，创建目录: {self.versions_dir}")
            self.versions_dir.mkdir(parents=True, exist_ok=True)
            return []
        
        # 扫描迁移文件
        migration_files = []
        for file_path in self.versions_dir.glob("v*.py"):
            if file_path.name != "__init__.py" and not file_path.name.startswith('.'):
                migration_files.append(file_path.stem)
        
        # 按文件名排序（版本号排序）
        migration_files.sort()
        
        logger.info(f"发现 {len(migration_files)} 个迁移文件: {migration_files}")
        return migration_files
    
    def load_migration_from_file(self, module_name: str) -> List[Migration]:
        """从文件加载迁移"""
        migrations = []
        
        try:
            # 构建完整的模块路径
            full_module_name = f"migrations.versions.{module_name}"
            
            # 动态导入模块
            if full_module_name in sys.modules:
                # 如果模块已经导入，重新加载
                module = importlib.reload(sys.modules[full_module_name])
            else:
                module = importlib.import_module(full_module_name)
            
            # 查找Migration类的子类
            for attr_name in dir(module):
                attr = getattr(module, attr_name)
                
                # 检查是否是Migration的子类（但不是Migration本身）
                if (inspect.isclass(attr) and 
                    issubclass(attr, Migration) and 
                    attr is not Migration):
                    
                    try:
                        # 实例化迁移
                        migration_instance = attr()
                        migrations.append(migration_instance)
                        logger.debug(f"从 {module_name} 加载迁移: {migration_instance.version}")
                        
                    except Exception as e:
                        logger.error(f"实例化迁移 {attr_name} 失败: {e}")
                        continue
            
            if not migrations:
                logger.warning(f"在文件 {module_name} 中未找到有效的迁移类")
                
        except Exception as e:
            logger.error(f"加载迁移文件 {module_name} 失败: {e}")
            logger.error(f"错误详情: {type(e).__name__}: {str(e)}")
        
        return migrations
    
    def load_all_migrations(self) -> List[Migration]:
        """加载所有迁移"""
        logger.info("🔍 开始扫描和加载迁移文件")
        
        migration_files = self.discover_migration_files()
        all_migrations = []
        
        for module_name in migration_files:
            migrations = self.load_migration_from_file(module_name)
            all_migrations.extend(migrations)
        
        # 按版本号排序
        all_migrations.sort(key=lambda m: self._version_to_tuple(m.version))
        
        self.loaded_migrations = all_migrations
        logger.info(f"✅ 成功加载 {len(all_migrations)} 个迁移")
        
        return all_migrations
    
    def _version_to_tuple(self, version: str) -> tuple:
        """将版本号转换为可比较的元组"""
        try:
            return tuple(map(int, version.split('.')))
        except:
            # 如果不是标准版本号格式，按字符串排序
            return (version,)
    
    def register_migrations(self, manager=None) -> int:
        """注册迁移到管理器"""
        if manager is None:
            manager = migration_manager
        
        registered_count = 0
        
        for migration in self.loaded_migrations:
            try:
                manager.register(migration)
                registered_count += 1
                logger.debug(f"注册迁移: {migration.version}")
                
            except Exception as e:
                logger.error(f"注册迁移 {migration.version} 失败: {e}")
                continue
        
        logger.info(f"✅ 成功注册 {registered_count} 个迁移到管理器")
        return registered_count
    
    def validate_migrations(self) -> bool:
        """验证迁移的完整性"""
        logger.info("🔍 验证迁移完整性")
        
        if not self.loaded_migrations:
            logger.warning("没有加载任何迁移")
            return True
        
        # 检查版本号重复
        versions = [m.version for m in self.loaded_migrations]
        if len(versions) != len(set(versions)):
            duplicates = [v for v in versions if versions.count(v) > 1]
            logger.error(f"发现重复的迁移版本: {duplicates}")
            return False
        
        # 检查依赖关系
        version_set = set(versions)
        for migration in self.loaded_migrations:
            for dep in migration.dependencies:
                if dep not in version_set:
                    logger.error(f"迁移 {migration.version} 依赖的版本 {dep} 不存在")
                    return False
        
        logger.info("✅ 迁移完整性验证通过")
        return True
    
    def get_migration_info(self) -> List[dict]:
        """获取迁移信息"""
        return [
            {
                "version": m.version,
                "description": m.description,
                "dependencies": m.dependencies,
                "checksum": m.get_checksum(),
                "class_name": m.__class__.__name__
            }
            for m in self.loaded_migrations
        ]

# 全局加载器实例
migration_loader = MigrationLoader()

def load_migrations(manager=None) -> int:
    """
    便捷函数：加载并注册所有迁移
    
    Args:
        manager: 迁移管理器实例，默认使用全局实例
        
    Returns:
        int: 成功注册的迁移数量
    """
    try:
        # 加载所有迁移
        migrations = migration_loader.load_all_migrations()
        
        # 验证迁移完整性
        if not migration_loader.validate_migrations():
            logger.error("迁移验证失败，停止注册")
            return 0
        
        # 注册到管理器
        registered_count = migration_loader.register_migrations(manager)
        
        logger.info(f"🎉 迁移加载完成！成功加载并注册 {registered_count} 个迁移")
        return registered_count
        
    except Exception as e:
        logger.error(f"加载迁移过程中发生异常: {e}")
        return 0

def reload_migrations(manager=None) -> int:
    """
    重新加载所有迁移（开发环境使用）
    
    Args:
        manager: 迁移管理器实例，默认使用全局实例
        
    Returns:
        int: 成功注册的迁移数量
    """
    if manager is None:
        manager = migration_manager
    
    # 清空现有迁移
    manager.migrations.clear()
    migration_loader.loaded_migrations.clear()
    
    logger.info("🔄 重新加载迁移")
    return load_migrations(manager)
