"""
迁移管理器 - 核心迁移逻辑
负责管理和执行数据库迁移的核心组件
"""
import asyncio
from typing import List, Dict, Any, Optional
from datetime import datetime
from sqlalchemy import text
from loguru import logger
import json
import traceback

from .base import Migration, MigrationError, version_to_tuple

class MigrationManager:
    """迁移管理器
    
    负责管理所有数据库迁移的执行、回滚和状态跟踪
    """
    
    def __init__(self):
        self.migrations: List[Migration] = []
        self._migration_table_created = False
        self._db_manager = None
    
    def set_db_manager(self, db_manager):
        """设置数据库管理器"""
        self._db_manager = db_manager
    
    async def _get_db_manager(self):
        """获取数据库管理器"""
        if self._db_manager is None:
            # 动态导入避免循环依赖
            from database import db_manager
            self._db_manager = db_manager
        return self._db_manager
    
    async def _ensure_migration_table(self):
        """确保迁移记录表存在"""
        if self._migration_table_created:
            return
            
        try:
            db = await self._get_db_manager()

            # 分别执行每个SQL语句，避免多命令问题
            await db.execute(text("""
                CREATE TABLE IF NOT EXISTS schema_migrations (
                    id SERIAL PRIMARY KEY,
                    version VARCHAR(50) UNIQUE NOT NULL,
                    description TEXT NOT NULL,
                    executed_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    execution_time_ms INTEGER,
                    checksum VARCHAR(64),
                    rollback_sql TEXT,
                    created_by VARCHAR(100) DEFAULT 'system',
                    CONSTRAINT unique_version UNIQUE (version)
                )
            """))

            await db.execute(text("""
                CREATE INDEX IF NOT EXISTS idx_schema_migrations_version
                ON schema_migrations(version)
            """))

            await db.execute(text("""
                CREATE INDEX IF NOT EXISTS idx_schema_migrations_executed_at
                ON schema_migrations(executed_at)
            """))

            # 添加表和字段注释
            await db.execute(text("""
                COMMENT ON TABLE schema_migrations IS '数据库schema迁移记录表'
            """))

            await db.execute(text("""
                COMMENT ON COLUMN schema_migrations.version IS '迁移版本号'
            """))

            await db.execute(text("""
                COMMENT ON COLUMN schema_migrations.description IS '迁移描述'
            """))

            await db.execute(text("""
                COMMENT ON COLUMN schema_migrations.executed_at IS '执行时间'
            """))

            await db.execute(text("""
                COMMENT ON COLUMN schema_migrations.execution_time_ms IS '执行耗时(毫秒)'
            """))

            await db.execute(text("""
                COMMENT ON COLUMN schema_migrations.checksum IS '迁移文件校验和'
            """))

            await db.execute(text("""
                COMMENT ON COLUMN schema_migrations.rollback_sql IS '回滚SQL语句'
            """))
            
            self._migration_table_created = True
            logger.info("✅ 迁移记录表已准备就绪")
            
        except Exception as e:
            logger.error(f"❌ 创建迁移记录表失败: {e}")
            raise MigrationError(f"无法创建迁移记录表: {e}")
    
    def register(self, migration: Migration):
        """注册迁移"""
        # 检查版本号重复
        existing_versions = [m.version for m in self.migrations]
        if migration.version in existing_versions:
            raise MigrationError(f"迁移版本 {migration.version} 已存在")
        
        self.migrations.append(migration)
        # 按版本号排序
        self.migrations.sort(key=lambda m: version_to_tuple(m.version))
        logger.debug(f"注册迁移: {migration}")
    
    async def get_executed_migrations(self) -> List[str]:
        """获取已执行的迁移版本列表"""
        await self._ensure_migration_table()
        
        try:
            db = await self._get_db_manager()
            result = await db.fetch_all(
                text("SELECT version FROM schema_migrations ORDER BY executed_at")
            )
            return [row['version'] for row in result]
        except Exception as e:
            logger.error(f"获取已执行迁移列表失败: {e}")
            return []
    
    async def get_pending_migrations(self) -> List[Migration]:
        """获取待执行的迁移"""
        executed = await self.get_executed_migrations()
        return [m for m in self.migrations if m.version not in executed]
    
    async def is_migration_executed(self, version: str) -> bool:
        """检查指定迁移是否已执行"""
        executed = await self.get_executed_migrations()
        return version in executed

    async def create_backup_point(self, migration_version: str) -> str:
        """创建备份点"""
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        backup_name = f"backup_before_{migration_version}_{timestamp}"

        try:
            db = await self._get_db_manager()
            # 记录备份点信息到system_config表
            from database import system_config_table

            backup_data = json.dumps({
                "timestamp": timestamp,
                "backup_name": backup_name,
                "migration_version": migration_version,
                "created_at": datetime.now().isoformat()
            })

            # 使用SQLAlchemy的insert语句
            query = system_config_table.insert().values(
                config_key=f"backup_point_{migration_version}",
                config_value=backup_data,
                description=f"迁移 {migration_version} 前的备份点"
            )

            try:
                await db.execute(query)
            except Exception as insert_error:
                # 如果插入失败（可能是重复键），尝试更新
                update_query = system_config_table.update().where(
                    system_config_table.c.config_key == f"backup_point_{migration_version}"
                ).values(
                    config_value=backup_data,
                    updated_at=text("CURRENT_TIMESTAMP")
                )
                await db.execute(update_query)

            logger.info(f"✅ 备份点已创建: {backup_name}")
            return backup_name

        except Exception as e:
            logger.error(f"❌ 创建备份点失败: {e}")
            raise MigrationError(f"无法创建备份点: {e}")

    async def execute_migration(self, migration: Migration) -> bool:
        """执行单个迁移"""
        logger.info(f"🚀 开始执行迁移: {migration}")

        start_time = datetime.now()
        db = await self._get_db_manager()

        try:
            # 1. 检查是否已执行
            if await self.is_migration_executed(migration.version):
                logger.warning(f"⚠️ 迁移 {migration.version} 已执行，跳过")
                return True

            # 2. 迁移前验证
            logger.info(f"🔍 执行迁移前验证: {migration.version}")
            if not await migration.validate_before_up(db):
                raise MigrationError(f"迁移前验证失败: {migration.version}")

            # 3. 创建备份点
            backup_name = await self.create_backup_point(migration.version)

            # 4. 在事务中执行迁移
            async with db.transaction():
                logger.info(f"⚡ 执行迁移操作: {migration.version}")

                # 执行迁移
                success = await migration.up(db)
                if not success:
                    raise MigrationError(f"迁移执行返回失败: {migration.version}")

                # 迁移后验证
                logger.info(f"✅ 执行迁移后验证: {migration.version}")
                if not await migration.validate_after_up(db):
                    raise MigrationError(f"迁移后验证失败: {migration.version}")

                # 获取回滚SQL（如果有）
                rollback_sql = await migration.get_rollback_sql(db)

                # 记录迁移执行
                execution_time = (datetime.now() - start_time).total_seconds() * 1000

                # 使用原始SQL字符串插入迁移记录
                insert_sql = """
                    INSERT INTO schema_migrations
                    (version, description, execution_time_ms, checksum, rollback_sql, created_by)
                    VALUES (:version, :description, :execution_time, :checksum, :rollback_sql, :created_by)
                """
                insert_values = {
                    "version": migration.version,
                    "description": migration.description,
                    "execution_time": int(execution_time),
                    "checksum": migration.get_checksum(),
                    "rollback_sql": rollback_sql,
                    "created_by": "migration_manager"
                }
                await db.execute(insert_sql, insert_values)

            logger.info(f"🎉 迁移执行成功: {migration.version} (耗时: {execution_time:.0f}ms)")
            return True

        except Exception as e:
            logger.error(f"❌ 迁移执行失败: {migration.version}")
            logger.error(f"错误详情: {e}")
            logger.error(f"堆栈跟踪: {traceback.format_exc()}")

            # 处理迁移失败
            await self._handle_migration_failure(migration, str(e))
            return False

    async def _handle_migration_failure(self, migration: Migration, error: str):
        """处理迁移失败"""
        logger.warning(f"⚠️ 处理迁移失败: {migration.version}")

        try:
            db = await self._get_db_manager()
            # 尝试执行回滚
            logger.info(f"🔄 尝试回滚迁移: {migration.version}")
            rollback_success = await migration.down(db)

            if rollback_success:
                logger.info(f"✅ 迁移回滚成功: {migration.version}")
            else:
                logger.error(f"❌ 迁移回滚失败: {migration.version}")

        except Exception as rollback_error:
            logger.error(f"❌ 回滚过程中发生错误: {rollback_error}")

    async def migrate_to_latest(self) -> Dict[str, Any]:
        """迁移到最新版本"""
        logger.info("🚀 开始数据库迁移到最新版本")

        await self._ensure_migration_table()

        pending_migrations = await self.get_pending_migrations()

        if not pending_migrations:
            logger.info("✅ 数据库已是最新版本，无需迁移")
            return {
                "success": True,
                "message": "数据库已是最新版本",
                "executed_migrations": 0,
                "total_migrations": len(self.migrations)
            }

        logger.info(f"📋 发现 {len(pending_migrations)} 个待执行迁移")

        executed_count = 0
        failed_migrations = []

        for migration in pending_migrations:
            try:
                success = await self.execute_migration(migration)
                if success:
                    executed_count += 1
                else:
                    failed_migrations.append(migration.version)
                    # 如果有迁移失败，停止后续迁移
                    break

            except Exception as e:
                logger.error(f"❌ 迁移 {migration.version} 执行异常: {e}")
                failed_migrations.append(migration.version)
                break

        # 返回迁移结果
        result = {
            "success": len(failed_migrations) == 0,
            "executed_migrations": executed_count,
            "total_pending": len(pending_migrations),
            "failed_migrations": failed_migrations
        }

        if result["success"]:
            logger.info(f"🎉 数据库迁移完成！执行了 {executed_count} 个迁移")
            result["message"] = f"成功执行 {executed_count} 个迁移"
        else:
            logger.error(f"❌ 数据库迁移失败！{len(failed_migrations)} 个迁移失败")
            result["message"] = f"迁移失败，{len(failed_migrations)} 个迁移未能执行"

        return result

    async def get_migration_status(self) -> Dict[str, Any]:
        """获取迁移状态信息"""
        await self._ensure_migration_table()

        executed_migrations = await self.get_executed_migrations()
        pending_migrations = await self.get_pending_migrations()

        return {
            "total_migrations": len(self.migrations),
            "executed_count": len(executed_migrations),
            "pending_count": len(pending_migrations),
            "executed_migrations": executed_migrations,
            "pending_migrations": [
                {
                    "version": m.version,
                    "description": m.description,
                    "dependencies": m.dependencies,
                    "checksum": m.get_checksum()
                } for m in pending_migrations
            ],
            "is_up_to_date": len(pending_migrations) == 0
        }

# 全局迁移管理器实例
migration_manager = MigrationManager()
