"""
迁移系统的数据库表结构定义
定义用于跟踪迁移历史的schema_migrations表
"""
from sqlalchemy import Column, Integer, String, Text, DateTime, Index
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.sql import func

Base = declarative_base()

class SchemaMigration(Base):
    """迁移记录表模型"""
    __tablename__ = 'schema_migrations'
    
    id = Column(Integer, primary_key=True, autoincrement=True, comment='主键ID')
    version = Column(String(50), unique=True, nullable=False, comment='迁移版本号')
    description = Column(Text, nullable=False, comment='迁移描述')
    executed_at = Column(DateTime, nullable=False, default=func.current_timestamp(), comment='执行时间')
    execution_time_ms = Column(Integer, comment='执行耗时(毫秒)')
    checksum = Column(String(64), comment='迁移文件校验和')
    rollback_sql = Column(Text, comment='回滚SQL语句')
    created_by = Column(String(100), default='system', comment='创建者')
    
    # 创建索引
    __table_args__ = (
        Index('idx_schema_migrations_version', 'version'),
        Index('idx_schema_migrations_executed_at', 'executed_at'),
        {'comment': '数据库schema迁移记录表'}
    )
    
    def __repr__(self):
        return f"<SchemaMigration(version='{self.version}', description='{self.description}')>"

# 迁移表的创建SQL（用于手动创建或验证）
CREATE_MIGRATION_TABLE_SQL = """
CREATE TABLE IF NOT EXISTS schema_migrations (
    id SERIAL PRIMARY KEY,
    version VARCHAR(50) UNIQUE NOT NULL,
    description TEXT NOT NULL,
    executed_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    execution_time_ms INTEGER,
    checksum VARCHAR(64),
    rollback_sql TEXT,
    created_by VARCHAR(100) DEFAULT 'system',
    CONSTRAINT unique_version UNIQUE (version)
);

CREATE INDEX IF NOT EXISTS idx_schema_migrations_version 
ON schema_migrations(version);

CREATE INDEX IF NOT EXISTS idx_schema_migrations_executed_at 
ON schema_migrations(executed_at);

COMMENT ON TABLE schema_migrations IS '数据库schema迁移记录表';
COMMENT ON COLUMN schema_migrations.id IS '主键ID';
COMMENT ON COLUMN schema_migrations.version IS '迁移版本号';
COMMENT ON COLUMN schema_migrations.description IS '迁移描述';
COMMENT ON COLUMN schema_migrations.executed_at IS '执行时间';
COMMENT ON COLUMN schema_migrations.execution_time_ms IS '执行耗时(毫秒)';
COMMENT ON COLUMN schema_migrations.checksum IS '迁移文件校验和';
COMMENT ON COLUMN schema_migrations.rollback_sql IS '回滚SQL语句';
COMMENT ON COLUMN schema_migrations.created_by IS '创建者';
"""

# 查询已执行迁移的SQL
GET_EXECUTED_MIGRATIONS_SQL = """
SELECT version, description, executed_at, execution_time_ms
FROM schema_migrations 
ORDER BY executed_at ASC;
"""

# 检查迁移是否已执行的SQL
CHECK_MIGRATION_EXECUTED_SQL = """
SELECT EXISTS (
    SELECT 1 FROM schema_migrations 
    WHERE version = %s
) as executed;
"""

# 插入迁移记录的SQL
INSERT_MIGRATION_RECORD_SQL = """
INSERT INTO schema_migrations 
(version, description, execution_time_ms, checksum, rollback_sql, created_by)
VALUES (%s, %s, %s, %s, %s, %s);
"""

# 删除迁移记录的SQL（用于回滚）
DELETE_MIGRATION_RECORD_SQL = """
DELETE FROM schema_migrations 
WHERE version = %s;
"""
