"""
版本 1.0.1: 为用户表添加头像字段
这是一个安全的迁移示例，展示如何添加新字段而不影响现有数据

迁移内容:
- 为users表添加avatar_url字段 (VARCHAR(500))
- 为现有用户设置默认头像
- 添加字段注释

安全措施:
- 使用IF NOT EXISTS确保幂等性
- 迁移前后都有验证
- 提供完整的回滚方法
"""
from migrations.base import Migration, MigrationError
from sqlalchemy import text
from loguru import logger

class AddUserAvatarMigration(Migration):
    """为用户表添加头像字段的迁移"""
    
    def __init__(self):
        super().__init__(
            version="1.0.1",
            description="为用户表添加头像字段",
            dependencies=[]  # 无依赖，这是第一个迁移
        )
    
    async def validate_before_up(self, db) -> bool:
        """迁移前验证：确保users表存在且没有avatar_url字段"""
        try:
            # 检查users表是否存在
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.tables 
                    WHERE table_schema = 'public' AND table_name = 'users'
                );
            """))
            
            if not result['exists']:
                logger.error("users表不存在，无法执行迁移")
                return False
            
            # 检查avatar_url字段是否已存在
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.columns 
                    WHERE table_schema = 'public' 
                    AND table_name = 'users' 
                    AND column_name = 'avatar_url'
                );
            """))
            
            if result['exists']:
                logger.warning("avatar_url字段已存在，跳过迁移")
                return False  # 字段已存在，不需要迁移
            
            # 检查是否有用户数据
            user_count = await db.fetch_one(text("SELECT COUNT(*) as count FROM users"))
            logger.info(f"当前用户数量: {user_count['count']}")
            
            logger.info("✅ 迁移前验证通过")
            return True
            
        except Exception as e:
            logger.error(f"迁移前验证失败: {e}")
            return False
    
    async def up(self, db) -> bool:
        """执行迁移：添加avatar_url字段"""
        try:
            logger.info("开始添加avatar_url字段到users表")
            
            # 1. 添加字段（使用IF NOT EXISTS确保安全）
            await db.execute(text("""
                ALTER TABLE users 
                ADD COLUMN IF NOT EXISTS avatar_url VARCHAR(500);
            """))
            logger.info("✅ avatar_url字段添加成功")
            
            # 2. 添加字段注释
            await db.execute(text("""
                COMMENT ON COLUMN users.avatar_url IS '用户头像URL地址，最大长度500字符';
            """))
            logger.info("✅ 字段注释添加成功")
            
            # 3. 为现有用户设置默认头像（可选）
            result = await db.execute(text("""
                UPDATE users 
                SET avatar_url = '/assets/default-avatar.png' 
                WHERE avatar_url IS NULL;
            """))
            logger.info(f"✅ 为现有用户设置默认头像，影响行数: {result}")
            
            # 4. 创建索引（如果需要按头像查询）
            await db.execute(text("""
                CREATE INDEX IF NOT EXISTS idx_users_avatar_url 
                ON users(avatar_url) 
                WHERE avatar_url IS NOT NULL;
            """))
            logger.info("✅ 头像字段索引创建成功")
            
            logger.info("🎉 avatar_url字段迁移完成")
            return True
            
        except Exception as e:
            logger.error(f"添加avatar_url字段失败: {e}")
            raise MigrationError(f"迁移执行失败: {e}", self.version, e)
    
    async def down(self, db) -> bool:
        """回滚迁移：移除avatar_url字段"""
        try:
            logger.info("开始回滚：从users表移除avatar_url字段")
            
            # 1. 删除索引
            await db.execute(text("""
                DROP INDEX IF EXISTS idx_users_avatar_url;
            """))
            logger.info("✅ 头像字段索引删除成功")
            
            # 2. 删除字段
            await db.execute(text("""
                ALTER TABLE users DROP COLUMN IF EXISTS avatar_url;
            """))
            logger.info("✅ avatar_url字段删除成功")
            
            logger.info("🔄 avatar_url字段回滚完成")
            return True
            
        except Exception as e:
            logger.error(f"回滚avatar_url字段失败: {e}")
            raise MigrationError(f"迁移回滚失败: {e}", self.version, e)
    
    async def validate_after_up(self, db) -> bool:
        """迁移后验证：确保字段已正确添加"""
        try:
            # 1. 验证字段是否存在
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.columns 
                    WHERE table_schema = 'public' 
                    AND table_name = 'users' 
                    AND column_name = 'avatar_url'
                );
            """))
            
            if not result['exists']:
                logger.error("avatar_url字段未成功添加")
                return False
            
            # 2. 验证字段类型和长度
            result = await db.fetch_one(text("""
                SELECT data_type, character_maximum_length, is_nullable
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = 'users' 
                AND column_name = 'avatar_url';
            """))
            
            if (result['data_type'] != 'character varying' or 
                result['character_maximum_length'] != 500 or
                result['is_nullable'] != 'YES'):
                logger.error(f"avatar_url字段属性不正确: {result}")
                return False
            
            # 3. 验证索引是否创建
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM pg_indexes 
                    WHERE tablename = 'users' 
                    AND indexname = 'idx_users_avatar_url'
                );
            """))
            
            if not result['exists']:
                logger.warning("avatar_url字段索引未创建，但不影响迁移")
            
            # 4. 验证现有用户是否有默认头像
            result = await db.fetch_one(text("""
                SELECT COUNT(*) as count 
                FROM users 
                WHERE avatar_url = '/assets/default-avatar.png';
            """))
            
            logger.info(f"设置默认头像的用户数量: {result['count']}")
            
            logger.info("✅ 迁移后验证通过")
            return True
            
        except Exception as e:
            logger.error(f"迁移后验证失败: {e}")
            return False
    
    async def get_rollback_sql(self, db) -> str:
        """获取回滚SQL语句"""
        return """
        -- 回滚迁移 v1.0.1: 删除用户头像字段
        DROP INDEX IF EXISTS idx_users_avatar_url;
        ALTER TABLE users DROP COLUMN IF EXISTS avatar_url;
        """
