"""
版本 1.0.2: 为用户表添加最后登录时间字段
这是第二个迁移示例，展示迁移系统的依赖管理和版本控制

迁移内容:
- 为users表添加last_login_at字段 (TIMESTAMP)
- 添加字段注释
- 创建索引以优化查询性能

安全措施:
- 依赖于v1.0.1迁移
- 使用IF NOT EXISTS确保幂等性
- 迁移前后都有验证
- 提供完整的回滚方法
"""
from migrations.base import Migration, MigrationError
from sqlalchemy import text
from loguru import logger

class AddUserLastLoginMigration(Migration):
    """为用户表添加最后登录时间字段的迁移"""
    
    def __init__(self):
        super().__init__(
            version="1.0.2",
            description="为用户表添加最后登录时间字段",
            dependencies=["1.0.1"]  # 依赖于头像字段迁移
        )
    
    async def validate_before_up(self, db) -> bool:
        """迁移前验证：确保users表存在且没有last_login_at字段"""
        try:
            # 检查users表是否存在
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.tables 
                    WHERE table_schema = 'public' AND table_name = 'users'
                );
            """))
            
            if not result['exists']:
                logger.error("users表不存在，无法执行迁移")
                return False
            
            # 检查依赖的avatar_url字段是否存在
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.columns 
                    WHERE table_schema = 'public' 
                    AND table_name = 'users' 
                    AND column_name = 'avatar_url'
                );
            """))
            
            if not result['exists']:
                logger.error("依赖的avatar_url字段不存在，无法执行迁移")
                return False
            
            # 检查last_login_at字段是否已存在
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.columns 
                    WHERE table_schema = 'public' 
                    AND table_name = 'users' 
                    AND column_name = 'last_login_at'
                );
            """))
            
            if result['exists']:
                logger.warning("last_login_at字段已存在，跳过迁移")
                return False  # 字段已存在，不需要迁移
            
            logger.info("✅ 迁移前验证通过")
            return True
            
        except Exception as e:
            logger.error(f"迁移前验证失败: {e}")
            return False
    
    async def up(self, db) -> bool:
        """执行迁移：添加last_login_at字段"""
        try:
            logger.info("开始添加last_login_at字段到users表")
            
            # 1. 添加字段（使用IF NOT EXISTS确保安全）
            await db.execute(text("""
                ALTER TABLE users 
                ADD COLUMN IF NOT EXISTS last_login_at TIMESTAMP;
            """))
            logger.info("✅ last_login_at字段添加成功")
            
            # 2. 添加字段注释
            await db.execute(text("""
                COMMENT ON COLUMN users.last_login_at IS '用户最后登录时间';
            """))
            logger.info("✅ 字段注释添加成功")
            
            # 3. 创建索引以优化查询性能
            await db.execute(text("""
                CREATE INDEX IF NOT EXISTS idx_users_last_login_at 
                ON users(last_login_at) 
                WHERE last_login_at IS NOT NULL;
            """))
            logger.info("✅ 最后登录时间索引创建成功")
            
            logger.info("🎉 last_login_at字段迁移完成")
            return True
            
        except Exception as e:
            logger.error(f"添加last_login_at字段失败: {e}")
            raise MigrationError(f"迁移执行失败: {e}", self.version, e)
    
    async def down(self, db) -> bool:
        """回滚迁移：移除last_login_at字段"""
        try:
            logger.info("开始回滚：从users表移除last_login_at字段")
            
            # 1. 删除索引
            await db.execute(text("""
                DROP INDEX IF EXISTS idx_users_last_login_at;
            """))
            logger.info("✅ 最后登录时间索引删除成功")
            
            # 2. 删除字段
            await db.execute(text("""
                ALTER TABLE users DROP COLUMN IF EXISTS last_login_at;
            """))
            logger.info("✅ last_login_at字段删除成功")
            
            logger.info("🔄 last_login_at字段回滚完成")
            return True
            
        except Exception as e:
            logger.error(f"回滚last_login_at字段失败: {e}")
            raise MigrationError(f"迁移回滚失败: {e}", self.version, e)
    
    async def validate_after_up(self, db) -> bool:
        """迁移后验证：确保字段已正确添加"""
        try:
            # 1. 验证字段是否存在
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.columns 
                    WHERE table_schema = 'public' 
                    AND table_name = 'users' 
                    AND column_name = 'last_login_at'
                );
            """))
            
            if not result['exists']:
                logger.error("last_login_at字段未成功添加")
                return False
            
            # 2. 验证字段类型
            result = await db.fetch_one(text("""
                SELECT data_type, is_nullable
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = 'users' 
                AND column_name = 'last_login_at';
            """))
            
            if (result['data_type'] != 'timestamp without time zone' or
                result['is_nullable'] != 'YES'):
                logger.error(f"last_login_at字段属性不正确: {result}")
                return False
            
            # 3. 验证索引是否创建
            result = await db.fetch_one(text("""
                SELECT EXISTS (
                    SELECT FROM pg_indexes 
                    WHERE tablename = 'users' 
                    AND indexname = 'idx_users_last_login_at'
                );
            """))
            
            if not result['exists']:
                logger.warning("last_login_at字段索引未创建，但不影响迁移")
            
            logger.info("✅ 迁移后验证通过")
            return True
            
        except Exception as e:
            logger.error(f"迁移后验证失败: {e}")
            return False
    
    async def get_rollback_sql(self, db) -> str:
        """获取回滚SQL语句"""
        return """
        -- 回滚迁移 v1.0.2: 删除用户最后登录时间字段
        DROP INDEX IF EXISTS idx_users_last_login_at;
        ALTER TABLE users DROP COLUMN IF EXISTS last_login_at;
        """
