"""
Pydantic 数据模型定义
用于 API 请求和响应的数据验证和序列化
"""

from pydantic import BaseModel, Field, validator
from typing import List, Optional, Any, Dict
from datetime import datetime
import json


# 基础模型
class BaseResponse(BaseModel):
    """基础响应模型"""
    success: bool = True
    message: str = "操作成功"
    data: Optional[Any] = None


class ErrorResponse(BaseModel):
    """错误响应模型"""
    success: bool = False
    message: str
    error: Optional[str] = None


# 用户相关模型
class UserBase(BaseModel):
    """用户基础模型"""
    phone: str = Field(..., min_length=1, max_length=20, description="手机号")
    name: str = Field(..., min_length=1, max_length=100, description="用户姓名")
    role: str = Field(default="user", description="用户角色")
    institutions: List[str] = Field(default=[], description="负责的机构ID列表")


class UserCreate(UserBase):
    """创建用户模型"""
    id: str = Field(..., min_length=1, max_length=50, description="用户ID")
    password: str = Field(..., min_length=1, max_length=255, description="密码")


class UserUpdate(BaseModel):
    """更新用户模型"""
    phone: Optional[str] = Field(None, min_length=1, max_length=20)
    name: Optional[str] = Field(None, min_length=1, max_length=100)
    password: Optional[str] = Field(None, min_length=1, max_length=255)
    role: Optional[str] = None
    institutions: Optional[List[str]] = None


class UserResponse(UserBase):
    """用户响应模型"""
    id: str
    created_at: datetime
    updated_at: datetime
    
    class Config:
        from_attributes = True


# 机构图片模型
class InstitutionImage(BaseModel):
    """机构图片模型"""
    id: str = Field(..., description="图片ID")
    url: str = Field(..., description="图片URL或Base64数据")
    uploadTime: datetime = Field(..., description="上传时间")


class InstitutionImageCreate(BaseModel):
    """创建机构图片模型"""
    id: str
    url: str


# 机构相关模型
class InstitutionBase(BaseModel):
    """机构基础模型"""
    name: str = Field(..., min_length=1, max_length=200, description="机构名称")
    institution_id: Optional[str] = Field(None, max_length=50, description="机构编号")
    owner_id: Optional[str] = Field(None, max_length=50, description="负责人ID")


class InstitutionCreate(InstitutionBase):
    """创建机构模型"""
    id: str = Field(..., min_length=1, max_length=50, description="机构ID")


class InstitutionUpdate(BaseModel):
    """更新机构模型"""
    name: Optional[str] = Field(None, min_length=1, max_length=200)
    institution_id: Optional[str] = Field(None, max_length=50)
    owner_id: Optional[str] = Field(None, max_length=50)


class InstitutionBatchCreate(BaseModel):
    """批量创建机构模型"""
    institutions: List[InstitutionCreate] = Field(..., description="机构列表")


class InstitutionBatchDelete(BaseModel):
    """批量删除机构模型"""
    institution_ids: List[str] = Field(..., description="要删除的机构ID列表")


class InstitutionBatchResponse(BaseModel):
    """批量操作响应模型"""
    success_count: int = Field(..., description="成功数量")
    error_count: int = Field(..., description="失败数量")
    errors: List[str] = Field(default=[], description="错误信息列表")
    message: str = Field(..., description="操作结果消息")


class InstitutionBatchCreate(BaseModel):
    """批量创建机构模型"""
    institutions: List[InstitutionCreate] = Field(..., description="机构列表")


class InstitutionBatchDelete(BaseModel):
    """批量删除机构模型"""
    institution_ids: List[str] = Field(..., description="要删除的机构ID列表")


class InstitutionBatchResponse(BaseModel):
    """批量操作响应模型"""
    success_count: int = Field(..., description="成功数量")
    error_count: int = Field(..., description="失败数量")
    errors: List[str] = Field(default=[], description="错误信息列表")
    message: str = Field(..., description="操作结果消息")


class InstitutionResponse(InstitutionBase):
    """机构响应模型"""
    id: str
    images: List[InstitutionImage] = []
    created_at: datetime
    updated_at: datetime
    
    class Config:
        from_attributes = True


# 系统配置模型
class SystemConfigItem(BaseModel):
    """系统配置项模型"""
    config_key: str = Field(..., max_length=100, description="配置键名")
    config_value: Any = Field(..., description="配置值")
    description: Optional[str] = Field(None, description="配置描述")


class SystemConfigResponse(BaseModel):
    """系统配置响应模型"""
    id: int
    config_key: str
    config_value: Any
    description: Optional[str]
    created_at: datetime
    updated_at: datetime
    
    class Config:
        from_attributes = True


# 历史统计数据模型
class UserStatsItem(BaseModel):
    """用户统计项模型"""
    userId: str
    userName: str
    institutionCount: int
    interactionScore: float
    performanceScore: float
    institutions: List[Dict[str, Any]]

    class Config:
        # 允许任意类型，提高兼容性
        arbitrary_types_allowed = True


class MonthlyHistoryCreate(BaseModel):
    """创建月度历史记录模型"""
    month: str = Field(..., pattern=r'^\d{4}-\d{2}$', description="月份，格式：YYYY-MM")
    save_time: datetime = Field(..., description="保存时间")
    total_users: int = Field(..., ge=0, description="总用户数")
    total_institutions: int = Field(..., ge=0, description="总机构数")
    total_images: int = Field(..., ge=0, description="总图片数")
    user_stats: List[UserStatsItem] = Field(..., description="用户统计数据")
    institutions_data: Optional[List[Dict[str, Any]]] = Field(None, description="机构图片数据")


class MonthlyHistoryResponse(BaseModel):
    """月度历史记录响应模型"""
    id: int
    month: str
    save_time: datetime
    total_users: int
    total_institutions: int
    total_images: int
    user_stats: List[UserStatsItem]
    institutions_data: Optional[List[Dict[str, Any]]] = Field(None, description="机构图片数据")
    created_at: datetime

    class Config:
        from_attributes = True


# 数据迁移模型
class MigrationData(BaseModel):
    """数据迁移模型"""
    users: List[Dict[str, Any]] = Field(..., description="用户数据")
    institutions: List[Dict[str, Any]] = Field(..., description="机构数据")
    systemConfig: Dict[str, Any] = Field(..., description="系统配置数据")
    historyData: Optional[Dict[str, Any]] = Field(None, description="历史统计数据")


class MigrationResponse(BaseModel):
    """数据迁移响应模型"""
    success: bool
    message: str
    migrated_counts: Dict[str, int]
    errors: List[str] = []


# 登录相关模型
class LoginRequest(BaseModel):
    """登录请求模型"""
    phone: str = Field(..., description="手机号")
    password: str = Field(..., description="密码")


class LoginResponse(BaseModel):
    """登录响应模型"""
    success: bool
    access_token: str
    refresh_token: str
    token_type: str = "bearer"
    expires_in: int  # token过期时间（秒）
    user: UserResponse
    message: str = "登录成功"


class TokenRefreshRequest(BaseModel):
    """Token刷新请求模型"""
    refresh_token: str = Field(..., description="刷新token")


class TokenRefreshResponse(BaseModel):
    """Token刷新响应模型"""
    success: bool
    access_token: str
    token_type: str = "bearer"
    expires_in: int  # token过期时间（秒）
    message: str = "Token刷新成功"
