"""
历史数据 API 路由
提供月度历史统计数据的 CRUD 操作接口
"""

from fastapi import APIRouter, HTTPException, Depends
from typing import List, Dict, Any
from loguru import logger
from datetime import datetime
from sqlalchemy import func

from database import monthly_history_table, get_database, DatabaseManager
from models import MonthlyHistoryCreate, MonthlyHistoryResponse, BaseResponse, UserResponse
from dependencies import get_current_active_user, require_admin

router = APIRouter()


@router.get("", response_model=BaseResponse, summary="获取所有历史记录")
async def get_all_history(
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """获取所有月度历史统计记录"""
    try:
        # 简单查询，让数据库处理字段选择
        query = monthly_history_table.select().order_by(monthly_history_table.c.month.desc())
        histories = await db.fetch_all(query)

        result = []
        for history in histories:
            try:
                # 安全地获取 institutions_data，确保兼容性
                institutions_data = []
                if hasattr(history, '_mapping') and "institutions_data" in history._mapping:
                    institutions_data = history["institutions_data"] or []
                elif hasattr(history, 'institutions_data'):
                    institutions_data = history.institutions_data or []

                result.append(MonthlyHistoryResponse(
                    id=history["id"],
                    month=history["month"],
                    save_time=history["save_time"],
                    total_users=history["total_users"],
                    total_institutions=history["total_institutions"],
                    total_images=history["total_images"],
                    user_stats=history["user_stats"],
                    institutions_data=institutions_data,
                    created_at=history["created_at"]
                ))
            except Exception as e:
                history_id = history.get("id", "unknown")
                logger.error(f"处理历史记录失败: {history_id}, 错误: {e}")
                # 跳过有问题的记录，继续处理其他记录
                continue

        return BaseResponse(
            success=True,
            message="获取历史记录成功",
            data=result
        )

    except Exception as e:
        logger.error(f"获取历史记录失败: {e}")
        # 如果是表不存在或字段不存在的错误，返回空列表而不是抛出异常
        if "does not exist" in str(e).lower():
            logger.info("历史记录表或字段不存在，返回空列表")
            return BaseResponse(
                success=True,
                message="获取历史记录成功",
                data=[]
            )
        raise HTTPException(status_code=500, detail="获取历史记录失败")


@router.get("/{month}", response_model=BaseResponse, summary="获取指定月份历史记录")
async def get_history_by_month(
    month: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """根据月份获取历史统计记录"""
    try:
        # 验证月份格式
        if not month or len(month) != 7 or month[4] != '-':
            raise HTTPException(status_code=400, detail="月份格式错误，应为 YYYY-MM")

        query = monthly_history_table.select().where(monthly_history_table.c.month == month)
        history = await db.fetch_one(query)

        if not history:
            raise HTTPException(status_code=404, detail="指定月份的历史记录不存在")

        # 安全地获取 institutions_data，确保兼容性
        institutions_data = history["institutions_data"] if "institutions_data" in history._mapping else []
        if institutions_data is None:
            institutions_data = []

        history_data = MonthlyHistoryResponse(
            id=history["id"],
            month=history["month"],
            save_time=history["save_time"],
            total_users=history["total_users"],
            total_institutions=history["total_institutions"],
            total_images=history["total_images"],
            user_stats=history["user_stats"],
            institutions_data=institutions_data,
            created_at=history["created_at"]
        )

        return BaseResponse(
            success=True,
            message="获取历史记录成功",
            data=history_data
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"获取指定月份历史记录失败: {e}")
        raise HTTPException(status_code=500, detail="获取历史记录失败")


@router.post("", response_model=BaseResponse, summary="保存月度历史记录")
async def save_monthly_history(
    history_data: Dict[str, Any],  # 改为更灵活的字典类型
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """保存月度历史统计记录"""
    try:
        logger.info(f"开始保存月度历史记录")
        logger.info(f"当前用户: {current_user.name} ({current_user.role})")
        logger.info(f"接收到的数据: {history_data}")

        # 提取必要字段
        month = history_data.get('month')
        save_time = history_data.get('save_time')
        total_users = history_data.get('total_users', 0)
        total_institutions = history_data.get('total_institutions', 0)
        total_images = history_data.get('total_images', 0)
        user_stats = history_data.get('user_stats', [])
        institutions_data = history_data.get('institutions_data', [])  # 新增：机构图片数据
        # 新增：保存类型标识，区分手动保存和月度重置
        save_type = history_data.get('save_type', 'manual')  # 'manual' 或 'monthly_reset'

        # 验证必要字段
        if not month:
            raise HTTPException(status_code=400, detail="缺少月份信息")
        if not save_time:
            raise HTTPException(status_code=400, detail="缺少保存时间")

        # 简化时间处理 - 使用naive datetime避免时区问题
        from datetime import datetime

        # 使用naive datetime（没有时区信息），让PostgreSQL使用服务器时区
        save_time = datetime.now().replace(microsecond=0)

        logger.info(f"使用naive datetime: {save_time}")
        logger.info(f"时间类型: {type(save_time)}, 时区: {save_time.tzinfo}")

        logger.info(f"处理月份: {month}, 用户数: {total_users}, 机构数: {total_institutions}, 保存类型: {save_type}")
        logger.info(f"处理后的save_time: {save_time}, 类型: {type(save_time)}, 时区: {save_time.tzinfo if hasattr(save_time, 'tzinfo') else 'N/A'}")

        # 检查该月份是否已有记录
        existing_history = await db.fetch_one(
            monthly_history_table.select().where(
                monthly_history_table.c.month == month
            )
        )

        # 保护机制：自动保存和月度重置不覆盖已有记录
        if save_type in ['monthly_reset', 'auto_save'] and existing_history:
            if save_type == 'monthly_reset':
                logger.info(f"月度重置跳过保存：{month} 月份已有数据，避免覆盖")
                return BaseResponse(message=f"{month} 月度记录已存在，跳过保存以保护现有数据")
            elif save_type == 'auto_save':
                logger.info(f"自动保存跳过保存：{month} 月份已有数据，避免覆盖")
                return BaseResponse(message=f"{month} 月度记录已存在，自动保存跳过以保护现有数据")

        # 确保user_stats是可序列化的
        import json
        if user_stats:
            try:
                # 尝试序列化以验证数据格式
                json.dumps(user_stats)
            except (TypeError, ValueError) as e:
                logger.error(f"user_stats数据无法序列化: {e}")
                user_stats = []  # 如果无法序列化，使用空数组

        # 使用SQLAlchemy方式，使用naive datetime
        import json

        if existing_history:
            # 只有手动保存才允许覆盖现有记录
            if save_type == 'manual':
                # 删除现有记录
                delete_query = monthly_history_table.delete().where(
                    monthly_history_table.c.month == month
                )
                await db.execute(delete_query)
                logger.info(f"手动保存：删除现有 {month} 月份记录")
            else:
                # 月度重置不应该到达这里，但为了安全起见
                logger.warning(f"月度重置尝试覆盖现有记录，已阻止")
                return BaseResponse(message=f"{month} 月度记录已存在，跳过保存")

        # 插入新记录 - 使用naive datetime对象
        query = monthly_history_table.insert().values(
            month=month,
            save_time=save_time,  # 使用naive datetime对象
            total_users=total_users,
            total_institutions=total_institutions,
            total_images=total_images,
            user_stats=user_stats,
            institutions_data=institutions_data
        )
        result = await db.execute(query)

        if existing_history and save_type == 'manual':
            message = f"{month} 月度记录覆盖保存成功"
        else:
            message = f"{month} 月度记录保存成功"

        logger.info(f"保存历史记录成功: {message}")

        return BaseResponse(message=message)

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"保存月度历史记录失败: {e}")
        logger.error(f"错误类型: {type(e)}")
        logger.error(f"错误详情: {str(e)}")
        import traceback
        logger.error(f"错误堆栈: {traceback.format_exc()}")
        raise HTTPException(status_code=500, detail=f"保存历史记录失败: {str(e)}")


@router.delete("", response_model=BaseResponse, summary="清空所有历史记录")
async def clear_all_history(
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """清空所有历史统计记录"""
    try:
        logger.info(f"管理员 {current_user.name} 请求清空所有历史记录")

        # 获取当前记录数
        count_query = "SELECT COUNT(*) as total FROM monthly_history"
        count_result = await db.fetch_one(count_query)
        total_records = count_result["total"] if count_result else 0

        if total_records == 0:
            return BaseResponse(message="没有历史记录需要清空")

        # 删除所有记录
        query = monthly_history_table.delete()
        await db.execute(query)

        logger.info(f"成功清空 {total_records} 条历史记录")
        return BaseResponse(message=f"成功清空 {total_records} 条历史记录")

    except Exception as e:
        logger.error(f"清空历史记录失败: {e}")
        raise HTTPException(status_code=500, detail="清空历史记录失败")


@router.delete("/{month}", response_model=BaseResponse, summary="删除指定月份历史记录")
async def delete_history_by_month(month: str, db: DatabaseManager = Depends(get_database)):
    """删除指定月份的历史统计记录"""
    try:
        # 验证月份格式
        if not month or len(month) != 7 or month[4] != '-':
            raise HTTPException(status_code=400, detail="月份格式错误，应为 YYYY-MM")

        # 检查记录是否存在
        existing_history = await db.fetch_one(
            monthly_history_table.select().where(monthly_history_table.c.month == month)
        )
        if not existing_history:
            raise HTTPException(status_code=404, detail="指定月份的历史记录不存在")

        # 删除记录
        query = monthly_history_table.delete().where(monthly_history_table.c.month == month)
        await db.execute(query)

        return BaseResponse(message=f"{month} 月度记录删除成功")

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"删除历史记录失败: {e}")
        raise HTTPException(status_code=500, detail="删除历史记录失败")


@router.get("/stats/summary", response_model=Dict[str, Any], summary="获取历史统计摘要")
async def get_history_summary(db: DatabaseManager = Depends(get_database)):
    """获取历史统计数据摘要"""
    try:
        # 获取总记录数
        total_query = "SELECT COUNT(*) as total FROM monthly_history"
        total_result = await db.fetch_one(total_query)
        total_records = total_result["total"] if total_result else 0
        
        # 获取最新记录
        latest_query = monthly_history_table.select().order_by(
            monthly_history_table.c.month.desc()
        ).limit(1)
        latest_record = await db.fetch_one(latest_query)
        
        # 获取最早记录
        earliest_query = monthly_history_table.select().order_by(
            monthly_history_table.c.month.asc()
        ).limit(1)
        earliest_record = await db.fetch_one(earliest_query)
        
        return {
            "total_records": total_records,
            "latest_month": latest_record["month"] if latest_record else None,
            "earliest_month": earliest_record["month"] if earliest_record else None,
            "latest_stats": {
                "total_users": latest_record["total_users"],
                "total_institutions": latest_record["total_institutions"],
                "total_images": latest_record["total_images"],
                "save_time": latest_record["save_time"]
            } if latest_record else None
        }
        
    except Exception as e:
        logger.error(f"获取历史统计摘要失败: {e}")
        raise HTTPException(status_code=500, detail="获取统计摘要失败")


@router.post("/cleanup", response_model=BaseResponse, summary="清理历史数据")
async def cleanup_old_history(
    keep_months: int = 12, 
    db: DatabaseManager = Depends(get_database)
):
    """清理旧的历史数据，保留最近N个月的记录"""
    try:
        if keep_months < 1:
            raise HTTPException(status_code=400, detail="保留月数必须大于0")
        
        # 获取需要保留的记录
        keep_query = monthly_history_table.select().order_by(
            monthly_history_table.c.month.desc()
        ).limit(keep_months)
        
        keep_records = await db.fetch_all(keep_query)
        
        if not keep_records:
            return BaseResponse(message="没有历史记录需要清理")
        
        # 获取最早需要保留的月份
        oldest_keep_month = keep_records[-1]["month"]
        
        # 删除更早的记录
        delete_query = monthly_history_table.delete().where(
            monthly_history_table.c.month < oldest_keep_month
        )
        
        deleted_count = await db.execute(delete_query)
        
        return BaseResponse(
            message=f"清理完成，删除了 {deleted_count} 条旧记录，保留最近 {keep_months} 个月的数据"
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"清理历史数据失败: {e}")
        raise HTTPException(status_code=500, detail="清理历史数据失败")




