"""
图片服务API
提供二进制图片的获取、上传、删除等功能
"""
import hashlib
import io
import uuid
from typing import Optional
from fastapi import APIRouter, Depends, HTTPException, UploadFile, File, Response
from fastapi.responses import StreamingResponse
from loguru import logger

from database import DatabaseManager, get_database
from dependencies import get_current_active_user, require_admin
from models import UserResponse

router = APIRouter(prefix="/api/images", tags=["图片服务"])

# 支持的图片格式
SUPPORTED_FORMATS = {'image/jpeg', 'image/png', 'image/webp'}
MAX_FILE_SIZE = 10 * 1024 * 1024  # 10MB

@router.get("/{image_id}")
async def get_image(
    image_id: str,
    db: DatabaseManager = Depends(get_database)
):
    """
    通过图片ID获取二进制图片数据
    
    Args:
        image_id: 图片唯一标识
        
    Returns:
        StreamingResponse: 图片二进制流
    """
    try:
        # 从数据库获取图片数据
        image_data = await db.fetch_one("""
            SELECT image_data, mime_type, original_filename, file_size
            FROM institution_images_binary 
            WHERE id = :image_id
        """, {"image_id": image_id})
        
        if not image_data:
            raise HTTPException(status_code=404, detail="图片不存在")
        
        # 处理文件名编码问题（避免中文字符导致的latin-1编码错误）
        filename = image_data['original_filename'] or image_id
        try:
            # 尝试编码为latin-1，如果失败则使用安全的文件名
            filename.encode('latin-1')
            safe_filename = filename
        except UnicodeEncodeError:
            # 如果包含非ASCII字符，使用图片ID作为文件名
            safe_filename = f"{image_id}.jpg"

        # 返回图片流
        return StreamingResponse(
            io.BytesIO(image_data['image_data']),
            media_type=image_data['mime_type'],
            headers={
                "Content-Disposition": f"inline; filename={safe_filename}",
                "Content-Length": str(image_data['file_size']),
                "Cache-Control": "public, max-age=31536000"  # 缓存1年
            }
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"获取图片失败: {e}")
        raise HTTPException(status_code=500, detail="获取图片失败")

@router.get("/{image_id}/info")
async def get_image_info(
    image_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """
    获取图片元数据信息
    
    Args:
        image_id: 图片唯一标识
        
    Returns:
        dict: 图片元数据
    """
    try:
        image_info = await db.fetch_one("""
            SELECT id, institution_id, original_filename, mime_type, 
                   file_size, width, height, compressed_quality, 
                   checksum, upload_time, created_at
            FROM institution_images_binary 
            WHERE id = :image_id
        """, {"image_id": image_id})
        
        if not image_info:
            raise HTTPException(status_code=404, detail="图片不存在")
        
        return {
            "success": True,
            "data": dict(image_info),
            "message": "获取图片信息成功"
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"获取图片信息失败: {e}")
        raise HTTPException(status_code=500, detail="获取图片信息失败")

@router.post("/upload")
async def upload_image(
    institution_id: str,
    file: UploadFile = File(...),
    quality: float = 0.8,
    max_width: int = 1200,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """
    上传并压缩图片
    
    Args:
        institution_id: 机构ID
        file: 上传的图片文件
        quality: 压缩质量 (0.1-1.0)
        max_width: 最大宽度
        
    Returns:
        dict: 上传结果
    """
    try:
        # 检查机构是否存在并验证权限
        from database import institutions_table
        institution = await db.fetch_one(
            institutions_table.select().where(institutions_table.c.id == institution_id)
        )

        if not institution:
            raise HTTPException(status_code=404, detail="机构不存在")

        # 权限检查：管理员可以上传到任何机构，普通用户只能上传到自己的机构
        if current_user.role != 'admin' and institution['owner_id'] != current_user.id:
            raise HTTPException(status_code=403, detail="权限不足，无法向此机构上传图片")

        # 验证文件类型
        if file.content_type not in SUPPORTED_FORMATS:
            raise HTTPException(
                status_code=400,
                detail=f"不支持的文件格式: {file.content_type}"
            )

        # 读取文件数据
        file_data = await file.read()

        # 验证文件大小
        if len(file_data) > MAX_FILE_SIZE:
            raise HTTPException(
                status_code=400,
                detail=f"文件过大，最大支持 {MAX_FILE_SIZE // 1024 // 1024}MB"
            )
        
        # 简化处理：直接使用原始数据（避免PIL依赖）
        compressed_data = file_data
        img_info = {'width': 800, 'height': 600}  # 默认尺寸
        
        # 计算校验和
        checksum = hashlib.md5(compressed_data).hexdigest()
        
        # 检查重复
        existing = await db.fetch_one("""
            SELECT id FROM institution_images_binary 
            WHERE checksum = :checksum
        """, {"checksum": checksum})
        
        if existing:
            return {
                "success": False,
                "message": "图片已存在",
                "data": {"existing_id": existing['id']}
            }
        
        # 生成图片ID
        image_id = str(uuid.uuid4())
        
        # 保存到数据库
        await db.execute("""
            INSERT INTO institution_images_binary 
            (id, institution_id, image_data, original_filename, mime_type, 
             file_size, width, height, compressed_quality, checksum)
            VALUES (:id, :institution_id, :image_data, :filename, :mime_type,
                    :file_size, :width, :height, :quality, :checksum)
        """, {
            "id": image_id,
            "institution_id": institution_id,
            "image_data": compressed_data,
            "filename": file.filename,
            "mime_type": file.content_type,
            "file_size": len(compressed_data),
            "width": img_info['width'],
            "height": img_info['height'],
            "quality": quality,
            "checksum": checksum
        })
        
        return {
            "success": True,
            "data": {
                "id": image_id,
                "url": f"/api/images/{image_id}",
                "size": len(compressed_data),
                "width": img_info['width'],
                "height": img_info['height']
            },
            "message": "图片上传成功"
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"图片上传失败: {e}")
        raise HTTPException(status_code=500, detail="图片上传失败")

@router.delete("/{image_id}")
async def delete_image(
    image_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """
    删除图片

    Args:
        image_id: 图片唯一标识

    Returns:
        dict: 删除结果
    """
    try:
        # 检查图片是否存在，并获取所属机构信息
        existing = await db.fetch_one("""
            SELECT ib.id, ib.institution_id, i.owner_id
            FROM institution_images_binary ib
            JOIN institutions i ON ib.institution_id = i.id
            WHERE ib.id = :image_id
        """, {"image_id": image_id})

        if not existing:
            raise HTTPException(status_code=404, detail="图片不存在")

        # 权限检查：管理员可以删除任何图片，普通用户只能删除自己机构的图片
        if current_user.role != 'admin' and existing['owner_id'] != current_user.id:
            raise HTTPException(status_code=403, detail="权限不足，无法删除此图片")

        # 删除图片
        await db.execute("""
            DELETE FROM institution_images_binary
            WHERE id = :image_id
        """, {"image_id": image_id})

        return {
            "success": True,
            "message": "图片删除成功"
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"图片删除失败: {e}")
        raise HTTPException(status_code=500, detail="图片删除失败")
