"""
机构管理 API 路由
提供机构和图片的 CRUD 操作接口
"""

from fastapi import APIRouter, HTTPException, Depends
from typing import List
from loguru import logger
from datetime import datetime

from database import (
    institutions_table, institution_images_table, 
    get_database, DatabaseManager
)
from models import (
    InstitutionCreate, InstitutionUpdate, InstitutionResponse,
    InstitutionImage, InstitutionImageCreate, BaseResponse, UserResponse,
    InstitutionBatchCreate, InstitutionBatchDelete, InstitutionBatchResponse
)
from dependencies import get_current_active_user, require_admin

router = APIRouter()


async def get_institution_with_images(institution_id: str, db: DatabaseManager):
    """获取机构及其图片信息"""
    # 获取机构基本信息
    inst_query = institutions_table.select().where(institutions_table.c.id == institution_id)
    institution = await db.fetch_one(inst_query)

    if not institution:
        return None

    # 优先从二进制图片表获取图片信息
    from database import institution_images_binary_table
    binary_images_query = institution_images_binary_table.select().where(
        institution_images_binary_table.c.institution_id == institution_id
    ).order_by(institution_images_binary_table.c.upload_time)

    binary_images = await db.fetch_all(binary_images_query)

    # 构建响应数据 - 只使用新表数据，不再回退到旧表
    institution_images = []

    # 只使用二进制表数据，返回API URL
    for img in binary_images:
        institution_images.append(InstitutionImage(
            id=img["id"],
            url=f"/api/images/{img['id']}",  # 使用图片API URL
            uploadTime=img["upload_time"]
        ))

    return InstitutionResponse(
        id=institution["id"],
        name=institution["name"],
        institution_id=institution["institution_id"],
        owner_id=institution["owner_id"],
        images=institution_images,
        created_at=institution["created_at"],
        updated_at=institution["updated_at"]
    )


@router.get("", response_model=BaseResponse, summary="获取机构列表")
async def get_all_institutions(
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """获取机构列表（管理员获取所有机构，普通用户只获取自己负责的机构）"""
    try:
        # 根据用户角色决定查询范围
        if current_user.role == 'admin':
            # 管理员获取所有机构
            query = institutions_table.select().order_by(institutions_table.c.created_at)
        else:
            # 普通用户只获取自己负责的机构
            query = institutions_table.select().where(
                institutions_table.c.owner_id == current_user.id
            ).order_by(institutions_table.c.created_at)

        institutions = await db.fetch_all(query)

        result = []
        for institution in institutions:
            inst_with_images = await get_institution_with_images(institution["id"], db)
            if inst_with_images:
                result.append(inst_with_images)

        logger.info(f"用户 {current_user.name}({current_user.role}) 获取到 {len(result)} 个机构")

        return BaseResponse(
            success=True,
            message="获取机构列表成功",
            data=result
        )

    except Exception as e:
        logger.error(f"获取机构列表失败: {e}")
        raise HTTPException(status_code=500, detail="获取机构列表失败")


@router.get("/{institution_id}", response_model=BaseResponse, summary="根据ID获取机构")
async def get_institution_by_id(
    institution_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """根据机构ID获取机构信息"""
    try:
        institution = await get_institution_with_images(institution_id, db)

        if not institution:
            raise HTTPException(status_code=404, detail="机构不存在")

        return BaseResponse(
            success=True,
            message="获取机构信息成功",
            data=institution
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"获取机构失败: {e}")
        raise HTTPException(status_code=500, detail="获取机构失败")


@router.get("/owner/{owner_id}", response_model=BaseResponse, summary="根据负责人ID获取机构")
async def get_institutions_by_owner(
    owner_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """根据负责人ID获取机构列表"""
    try:
        query = institutions_table.select().where(
            institutions_table.c.owner_id == owner_id
        ).order_by(institutions_table.c.created_at)

        institutions = await db.fetch_all(query)

        result = []
        for institution in institutions:
            inst_with_images = await get_institution_with_images(institution["id"], db)
            if inst_with_images:
                result.append(inst_with_images)

        return BaseResponse(
            success=True,
            message="获取机构列表成功",
            data=result
        )

    except Exception as e:
        logger.error(f"根据负责人获取机构失败: {e}")
        raise HTTPException(status_code=500, detail="获取机构失败")


@router.post("", response_model=BaseResponse, summary="创建机构")
async def create_institution(
    institution_data: InstitutionCreate,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """创建新机构"""
    try:
        # 检查机构ID是否已存在
        existing_inst = await db.fetch_one(
            institutions_table.select().where(institutions_table.c.id == institution_data.id)
        )
        if existing_inst:
            raise HTTPException(status_code=400, detail="机构ID已存在")

        # 检查机构编号是否已存在（如果提供了编号）
        if institution_data.institution_id:
            existing_inst_id = await db.fetch_one(
                institutions_table.select().where(
                    institutions_table.c.institution_id == institution_data.institution_id
                )
            )
            if existing_inst_id:
                raise HTTPException(status_code=400, detail="机构编号已存在")

        # 插入新机构
        query = institutions_table.insert().values(
            id=institution_data.id,
            name=institution_data.name,
            institution_id=institution_data.institution_id,
            owner_id=institution_data.owner_id
        )

        await db.execute(query)

        # 获取创建的机构信息
        created_institution = await get_institution_with_images(institution_data.id, db)

        return BaseResponse(
            success=True,
            message="创建机构成功",
            data=created_institution
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"创建机构失败: {e}")
        raise HTTPException(status_code=500, detail="创建机构失败")


@router.put("/{institution_id}", response_model=BaseResponse, summary="更新机构")
async def update_institution(
    institution_id: str,
    institution_data: InstitutionUpdate,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """更新机构信息"""
    try:
        # 检查机构是否存在
        existing_inst = await db.fetch_one(
            institutions_table.select().where(institutions_table.c.id == institution_id)
        )
        if not existing_inst:
            raise HTTPException(status_code=404, detail="机构不存在")

        # 构建更新数据
        update_data = {}
        if institution_data.name is not None:
            update_data["name"] = institution_data.name
        if institution_data.institution_id is not None:
            # 检查机构编号是否被其他机构使用
            inst_id_check = await db.fetch_one(
                institutions_table.select().where(
                    (institutions_table.c.institution_id == institution_data.institution_id) &
                    (institutions_table.c.id != institution_id)
                )
            )
            if inst_id_check:
                raise HTTPException(status_code=400, detail="机构编号已被其他机构使用")
            update_data["institution_id"] = institution_data.institution_id
        if institution_data.owner_id is not None:
            update_data["owner_id"] = institution_data.owner_id

        if not update_data:
            raise HTTPException(status_code=400, detail="没有提供更新数据")

        # 执行更新
        query = institutions_table.update().where(
            institutions_table.c.id == institution_id
        ).values(**update_data)
        await db.execute(query)

        # 获取更新后的机构信息
        updated_institution = await get_institution_with_images(institution_id, db)

        return BaseResponse(
            success=True,
            message="更新机构成功",
            data=updated_institution
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"更新机构失败: {e}")
        raise HTTPException(status_code=500, detail="更新机构失败")


@router.delete("/{institution_id}", response_model=BaseResponse, summary="删除机构")
async def delete_institution(
    institution_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """删除机构（级联删除相关图片）"""
    try:
        # 检查机构是否存在
        existing_inst = await db.fetch_one(
            institutions_table.select().where(institutions_table.c.id == institution_id)
        )
        if not existing_inst:
            raise HTTPException(status_code=404, detail="机构不存在")
        
        # 删除机构（外键约束会自动删除相关图片）
        query = institutions_table.delete().where(institutions_table.c.id == institution_id)
        await db.execute(query)
        
        return BaseResponse(message="机构删除成功")
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"删除机构失败: {e}")
        raise HTTPException(status_code=500, detail="删除机构失败")


# 图片管理相关接口
@router.post("/{institution_id}/images", response_model=BaseResponse, summary="添加机构图片")
async def add_institution_image(
    institution_id: str,
    image_data: InstitutionImageCreate,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """为机构添加图片"""
    try:
        logger.info(f"开始添加图片到机构 {institution_id}")
        logger.info(f"图片数据: id={image_data.id}, url长度={len(image_data.url)}")

        # 检查机构是否存在
        existing_inst = await db.fetch_one(
            institutions_table.select().where(institutions_table.c.id == institution_id)
        )
        if not existing_inst:
            logger.error(f"机构不存在: {institution_id}")
            raise HTTPException(status_code=404, detail="机构不存在")

        logger.info(f"找到机构: {existing_inst['name']}")

        # 检查图片ID是否已存在
        existing_image = await db.fetch_one(
            institution_images_table.select().where(institution_images_table.c.id == image_data.id)
        )
        if existing_image:
            logger.error(f"图片ID已存在: {image_data.id}")
            raise HTTPException(status_code=400, detail="图片ID已存在")

        # 使用当前时间作为上传时间
        upload_time = datetime.now()
        logger.info(f"使用当前时间作为上传时间: {upload_time}")

        # 插入图片记录
        query = institution_images_table.insert().values(
            id=image_data.id,
            institution_id=institution_id,
            url=image_data.url,
            upload_time=upload_time
        )

        logger.info("准备执行数据库插入操作")
        await db.execute(query)
        logger.info("图片记录插入成功")

        return BaseResponse(message="图片添加成功")

    except HTTPException as e:
        logger.error(f"HTTP异常: {e.detail}")
        raise
    except Exception as e:
        logger.error(f"添加机构图片失败: {type(e).__name__}: {str(e)}")
        logger.error(f"详细错误信息: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail=f"添加图片失败: {str(e)}")


@router.delete("/{institution_id}/images/{image_id}", response_model=BaseResponse, summary="删除机构图片")
async def delete_institution_image(
    institution_id: str,
    image_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """删除机构图片"""
    try:
        # 检查图片是否存在且属于指定机构
        existing_image = await db.fetch_one(
            institution_images_table.select().where(
                (institution_images_table.c.id == image_id) &
                (institution_images_table.c.institution_id == institution_id)
            )
        )
        if not existing_image:
            raise HTTPException(status_code=404, detail="图片不存在")

        # 删除图片记录
        query = institution_images_table.delete().where(
            institution_images_table.c.id == image_id
        )
        await db.execute(query)

        return BaseResponse(message="图片删除成功")

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"删除机构图片失败: {e}")
        raise HTTPException(status_code=500, detail="删除图片失败")


@router.get("/institution-id/{inst_id}", response_model=BaseResponse, summary="根据机构编号获取机构")
async def get_institution_by_institution_id(
    inst_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """根据机构编号获取机构信息"""
    try:
        query = institutions_table.select().where(institutions_table.c.institution_id == inst_id)
        institution = await db.fetch_one(query)

        if not institution:
            raise HTTPException(status_code=404, detail="机构不存在")

        institution_data = await get_institution_with_images(institution["id"], db)

        return BaseResponse(
            success=True,
            message="获取机构信息成功",
            data=institution_data
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"根据机构编号获取机构失败: {e}")
        raise HTTPException(status_code=500, detail="获取机构失败")


@router.post("/batch", response_model=InstitutionBatchResponse, summary="批量创建机构")
async def batch_create_institutions(
    batch_data: InstitutionBatchCreate,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """批量创建机构"""
    success_count = 0
    error_count = 0
    errors = []

    try:
        for i, institution_data in enumerate(batch_data.institutions):
            try:
                # 检查机构ID是否已存在
                existing_inst = await db.fetch_one(
                    institutions_table.select().where(institutions_table.c.id == institution_data.id)
                )
                if existing_inst:
                    errors.append(f"第{i+1}个机构: 机构ID {institution_data.id} 已存在")
                    error_count += 1
                    continue

                # 检查机构编号是否已存在（如果提供了编号）
                if institution_data.institution_id:
                    existing_inst_id = await db.fetch_one(
                        institutions_table.select().where(
                            institutions_table.c.institution_id == institution_data.institution_id
                        )
                    )
                    if existing_inst_id:
                        errors.append(f"第{i+1}个机构: 机构编号 {institution_data.institution_id} 已存在")
                        error_count += 1
                        continue

                # 插入新机构
                query = institutions_table.insert().values(
                    id=institution_data.id,
                    name=institution_data.name,
                    institution_id=institution_data.institution_id,
                    owner_id=institution_data.owner_id
                )

                await db.execute(query)
                success_count += 1

            except Exception as e:
                errors.append(f"第{i+1}个机构: {str(e)}")
                error_count += 1

        message = f"批量创建完成: 成功 {success_count} 个, 失败 {error_count} 个"
        logger.info(message)

        return InstitutionBatchResponse(
            success_count=success_count,
            error_count=error_count,
            errors=errors,
            message=message
        )

    except Exception as e:
        logger.error(f"批量创建机构失败: {e}")
        raise HTTPException(status_code=500, detail="批量创建机构失败")


@router.get("/batch/test", summary="测试批量删除路由")
async def test_batch_route():
    """测试批量删除路由是否可访问"""
    return {"message": "批量删除路由正常工作"}


@router.post("/batch-delete", response_model=InstitutionBatchResponse, summary="批量删除机构")
async def batch_delete_institutions(
    batch_data: InstitutionBatchDelete,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """批量删除机构"""
    success_count = 0
    error_count = 0
    errors = []

    try:
        logger.info(f"收到批量删除请求，机构ID列表: {batch_data.institution_ids}")
        for institution_id in batch_data.institution_ids:
            try:
                # 检查机构是否存在
                existing_inst = await db.fetch_one(
                    institutions_table.select().where(institutions_table.c.id == institution_id)
                )
                if not existing_inst:
                    errors.append(f"机构 {institution_id} 不存在")
                    error_count += 1
                    continue

                # 删除机构（外键约束会自动删除相关图片）
                query = institutions_table.delete().where(institutions_table.c.id == institution_id)
                await db.execute(query)
                success_count += 1

            except Exception as e:
                errors.append(f"删除机构 {institution_id} 失败: {str(e)}")
                error_count += 1

        message = f"批量删除完成: 成功 {success_count} 个, 失败 {error_count} 个"
        logger.info(message)

        return InstitutionBatchResponse(
            success_count=success_count,
            error_count=error_count,
            errors=errors,
            message=message
        )

    except Exception as e:
        logger.error(f"批量删除机构失败: {e}")
        raise HTTPException(status_code=500, detail="批量删除机构失败")


