"""
数据迁移 API 路由
提供 localStorage 数据迁移到数据库的功能
"""

from fastapi import APIRouter, HTTPException, Depends
from typing import List, Dict, Any
from loguru import logger
from datetime import datetime
import json

from database import (
    users_table, institutions_table, institution_images_table,
    system_config_table, monthly_history_table,
    get_database, DatabaseManager
)
from models import MigrationData, MigrationResponse, BaseResponse

router = APIRouter()


async def migrate_users(users_data: List[Dict[str, Any]], db: DatabaseManager) -> Dict[str, int]:
    """迁移用户数据"""
    migrated_count = 0
    error_count = 0
    errors = []
    
    try:
        for user in users_data:
            try:
                # 检查用户是否已存在
                existing_user = await db.fetch_one(
                    users_table.select().where(users_table.c.id == user.get("id"))
                )
                
                if existing_user:
                    # 更新现有用户
                    query = users_table.update().where(users_table.c.id == user.get("id")).values(
                        phone=user.get("phone", ""),
                        password=user.get("password", ""),
                        name=user.get("name", ""),
                        role=user.get("role", "user"),
                        institutions=user.get("institutions", [])
                    )
                    await db.execute(query)
                else:
                    # 创建新用户
                    query = users_table.insert().values(
                        id=user.get("id"),
                        phone=user.get("phone", ""),
                        password=user.get("password", ""),
                        name=user.get("name", ""),
                        role=user.get("role", "user"),
                        institutions=user.get("institutions", [])
                    )
                    await db.execute(query)
                
                migrated_count += 1
                
            except Exception as e:
                error_count += 1
                errors.append(f"用户 {user.get('id', 'unknown')} 迁移失败: {str(e)}")
                logger.error(f"用户迁移失败: {e}")
    
    except Exception as e:
        logger.error(f"用户数据迁移过程失败: {e}")
        raise
    
    return {
        "migrated": migrated_count,
        "errors": error_count,
        "error_messages": errors
    }


async def migrate_institutions(institutions_data: List[Dict[str, Any]], db: DatabaseManager) -> Dict[str, int]:
    """迁移机构数据"""
    migrated_count = 0
    image_count = 0
    error_count = 0
    errors = []
    
    try:
        for institution in institutions_data:
            try:
                inst_id = institution.get("id")
                
                # 检查机构是否已存在
                existing_inst = await db.fetch_one(
                    institutions_table.select().where(institutions_table.c.id == inst_id)
                )
                
                if existing_inst:
                    # 更新现有机构
                    query = institutions_table.update().where(institutions_table.c.id == inst_id).values(
                        name=institution.get("name", ""),
                        institution_id=institution.get("institutionId"),
                        owner_id=institution.get("ownerId")
                    )
                    await db.execute(query)
                else:
                    # 创建新机构
                    query = institutions_table.insert().values(
                        id=inst_id,
                        name=institution.get("name", ""),
                        institution_id=institution.get("institutionId"),
                        owner_id=institution.get("ownerId")
                    )
                    await db.execute(query)
                
                # 处理机构图片
                images = institution.get("images", [])
                if images:
                    # 先删除现有图片
                    delete_images_query = institution_images_table.delete().where(
                        institution_images_table.c.institution_id == inst_id
                    )
                    await db.execute(delete_images_query)
                    
                    # 插入新图片
                    for image in images:
                        try:
                            upload_time = image.get("uploadTime")
                            if isinstance(upload_time, str):
                                upload_time = datetime.fromisoformat(upload_time.replace('Z', '+00:00'))
                            elif not isinstance(upload_time, datetime):
                                upload_time = datetime.now()
                            
                            image_query = institution_images_table.insert().values(
                                id=image.get("id"),
                                institution_id=inst_id,
                                url=image.get("url", ""),
                                upload_time=upload_time
                            )
                            await db.execute(image_query)
                            image_count += 1
                            
                        except Exception as e:
                            errors.append(f"机构 {inst_id} 的图片 {image.get('id', 'unknown')} 迁移失败: {str(e)}")
                
                migrated_count += 1
                
            except Exception as e:
                error_count += 1
                errors.append(f"机构 {institution.get('id', 'unknown')} 迁移失败: {str(e)}")
                logger.error(f"机构迁移失败: {e}")
    
    except Exception as e:
        logger.error(f"机构数据迁移过程失败: {e}")
        raise
    
    return {
        "migrated": migrated_count,
        "images": image_count,
        "errors": error_count,
        "error_messages": errors
    }


async def migrate_system_config(config_data: Dict[str, Any], db: DatabaseManager) -> Dict[str, int]:
    """迁移系统配置数据"""
    migrated_count = 0
    error_count = 0
    errors = []
    
    try:
        for key, value in config_data.items():
            try:
                # 检查配置是否已存在
                existing_config = await db.fetch_one(
                    system_config_table.select().where(system_config_table.c.config_key == key)
                )
                
                if existing_config:
                    # 更新现有配置
                    query = system_config_table.update().where(
                        system_config_table.c.config_key == key
                    ).values(config_value=value)
                    await db.execute(query)
                else:
                    # 创建新配置
                    query = system_config_table.insert().values(
                        config_key=key,
                        config_value=value,
                        description=f"从localStorage迁移的配置项: {key}"
                    )
                    await db.execute(query)
                
                migrated_count += 1
                
            except Exception as e:
                error_count += 1
                errors.append(f"配置项 {key} 迁移失败: {str(e)}")
                logger.error(f"配置迁移失败: {e}")
    
    except Exception as e:
        logger.error(f"系统配置迁移过程失败: {e}")
        raise
    
    return {
        "migrated": migrated_count,
        "errors": error_count,
        "error_messages": errors
    }


async def migrate_history_data(history_data: Dict[str, Any], db: DatabaseManager) -> Dict[str, int]:
    """迁移历史统计数据"""
    migrated_count = 0
    error_count = 0
    errors = []
    
    try:
        if not history_data:
            return {"migrated": 0, "errors": 0, "error_messages": []}
        
        for month, data in history_data.items():
            try:
                # 解析保存时间
                save_time = data.get("saveTime")
                if isinstance(save_time, str):
                    save_time = datetime.fromisoformat(save_time.replace('Z', '+00:00'))
                elif not isinstance(save_time, datetime):
                    save_time = datetime.now()
                
                # 检查记录是否已存在
                existing_history = await db.fetch_one(
                    monthly_history_table.select().where(monthly_history_table.c.month == month)
                )
                
                if existing_history:
                    # 更新现有记录
                    query = monthly_history_table.update().where(
                        monthly_history_table.c.month == month
                    ).values(
                        save_time=save_time,
                        total_users=data.get("totalUsers", 0),
                        total_institutions=data.get("totalInstitutions", 0),
                        total_images=data.get("totalImages", 0),
                        user_stats=data.get("userStats", [])
                    )
                    await db.execute(query)
                else:
                    # 创建新记录
                    query = monthly_history_table.insert().values(
                        month=month,
                        save_time=save_time,
                        total_users=data.get("totalUsers", 0),
                        total_institutions=data.get("totalInstitutions", 0),
                        total_images=data.get("totalImages", 0),
                        user_stats=data.get("userStats", [])
                    )
                    await db.execute(query)
                
                migrated_count += 1
                
            except Exception as e:
                error_count += 1
                errors.append(f"历史记录 {month} 迁移失败: {str(e)}")
                logger.error(f"历史数据迁移失败: {e}")
    
    except Exception as e:
        logger.error(f"历史数据迁移过程失败: {e}")
        raise
    
    return {
        "migrated": migrated_count,
        "errors": error_count,
        "error_messages": errors
    }


@router.post("/migrate", response_model=MigrationResponse, summary="执行数据迁移")
async def migrate_data(
    migration_data: MigrationData,
    db: DatabaseManager = Depends(get_database)
):
    """执行完整的数据迁移过程"""
    try:
        logger.info("开始执行数据迁移...")
        
        all_errors = []
        migrated_counts = {
            "users": 0,
            "institutions": 0,
            "images": 0,
            "system_config": 0,
            "history": 0
        }
        
        async with db.transaction():
            # 迁移用户数据
            if migration_data.users:
                logger.info(f"开始迁移 {len(migration_data.users)} 个用户...")
                user_result = await migrate_users(migration_data.users, db)
                migrated_counts["users"] = user_result["migrated"]
                all_errors.extend(user_result["error_messages"])
            
            # 迁移机构数据
            if migration_data.institutions:
                logger.info(f"开始迁移 {len(migration_data.institutions)} 个机构...")
                inst_result = await migrate_institutions(migration_data.institutions, db)
                migrated_counts["institutions"] = inst_result["migrated"]
                migrated_counts["images"] = inst_result["images"]
                all_errors.extend(inst_result["error_messages"])
            
            # 迁移系统配置
            if migration_data.systemConfig:
                logger.info(f"开始迁移 {len(migration_data.systemConfig)} 个配置项...")
                config_result = await migrate_system_config(migration_data.systemConfig, db)
                migrated_counts["system_config"] = config_result["migrated"]
                all_errors.extend(config_result["error_messages"])
            
            # 迁移历史数据
            if migration_data.historyData:
                logger.info(f"开始迁移 {len(migration_data.historyData)} 个历史记录...")
                history_result = await migrate_history_data(migration_data.historyData, db)
                migrated_counts["history"] = history_result["migrated"]
                all_errors.extend(history_result["error_messages"])
        
        logger.info("数据迁移完成")
        
        return MigrationResponse(
            success=True,
            message="数据迁移完成",
            migrated_counts=migrated_counts,
            errors=all_errors
        )
        
    except Exception as e:
        logger.error(f"数据迁移失败: {e}")
        return MigrationResponse(
            success=False,
            message=f"数据迁移失败: {str(e)}",
            migrated_counts={},
            errors=[str(e)]
        )


@router.post("/check", response_model=Dict[str, Any], summary="检查迁移状态")
async def check_migration_status(db: DatabaseManager = Depends(get_database)):
    """检查数据库中的数据状态，用于判断是否需要迁移"""
    try:
        # 检查各表的记录数
        users_count = await db.fetch_one("SELECT COUNT(*) as count FROM users")
        institutions_count = await db.fetch_one("SELECT COUNT(*) as count FROM institutions")
        images_count = await db.fetch_one("SELECT COUNT(*) as count FROM institution_images")
        config_count = await db.fetch_one("SELECT COUNT(*) as count FROM system_config")
        history_count = await db.fetch_one("SELECT COUNT(*) as count FROM monthly_history")
        
        return {
            "database_status": "connected",
            "data_counts": {
                "users": users_count["count"] if users_count else 0,
                "institutions": institutions_count["count"] if institutions_count else 0,
                "images": images_count["count"] if images_count else 0,
                "system_config": config_count["count"] if config_count else 0,
                "history": history_count["count"] if history_count else 0
            },
            "migration_needed": (
                (users_count["count"] if users_count else 0) <= 1 and  # 只有默认管理员
                (institutions_count["count"] if institutions_count else 0) == 0
            ),
            "timestamp": datetime.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"检查迁移状态失败: {e}")
        raise HTTPException(status_code=500, detail="检查迁移状态失败")


@router.post("/clear", response_model=BaseResponse, summary="清空数据库数据")
async def clear_database(
    confirm: bool = False,
    db: DatabaseManager = Depends(get_database)
):
    """清空数据库中的所有数据（危险操作，需要确认）"""
    if not confirm:
        raise HTTPException(status_code=400, detail="请设置 confirm=true 确认清空操作")
    
    try:
        async with db.transaction():
            # 按依赖关系顺序删除数据
            await db.execute(institution_images_table.delete())
            await db.execute(institutions_table.delete())
            await db.execute(monthly_history_table.delete())
            await db.execute(system_config_table.delete())
            await db.execute(users_table.delete())
            
            # 重新插入默认数据
            await db.execute(
                users_table.insert().values(
                    id="admin",
                    phone="admin",
                    password="admin123",
                    name="系统管理员",
                    role="admin",
                    institutions=[]
                )
            )
            
            # 插入默认系统配置
            default_configs = [
                ("initialized", True, "系统是否已初始化"),
                ("version", "8.8.0", "系统版本"),
                ("hasDefaultData", False, "是否有默认示例数据")
            ]
            
            for key, value, desc in default_configs:
                await db.execute(
                    system_config_table.insert().values(
                        config_key=key,
                        config_value=value,
                        description=desc
                    )
                )
        
        logger.info("数据库已清空并重置为默认状态")
        return BaseResponse(message="数据库已清空并重置为默认状态")
        
    except Exception as e:
        logger.error(f"清空数据库失败: {e}")
        raise HTTPException(status_code=500, detail="清空数据库失败")


# ==================== Schema 迁移管理 API ====================

@router.get("/schema/status", summary="获取数据库schema状态")
async def get_schema_status(
    db: DatabaseManager = Depends(get_database)
):
    """获取数据库schema迁移状态"""
    try:
        from migrations.manager import migration_manager

        status = await migration_manager.get_migration_status()

        return {
            "success": True,
            "data": status,
            "message": "获取schema状态成功"
        }

    except Exception as e:
        logger.error(f"获取schema状态失败: {e}")
        raise HTTPException(status_code=500, detail=f"获取schema状态失败: {str(e)}")


@router.post("/schema/migrate", summary="手动执行数据库schema迁移")
async def manual_migrate_schema(
    db: DatabaseManager = Depends(get_database)
):
    """手动执行数据库schema迁移到最新版本"""
    try:
        from migrations.manager import migration_manager

        logger.info("开始手动执行数据库schema迁移")
        result = await migration_manager.migrate_to_latest()

        if result["success"]:
            return {
                "success": True,
                "data": {
                    "executed_migrations": result["executed_migrations"],
                    "total_pending": result.get("total_pending", 0),
                    "message": result["message"]
                },
                "message": "数据库迁移成功"
            }
        else:
            raise HTTPException(
                status_code=500,
                detail={
                    "message": result["message"],
                    "failed_migrations": result.get("failed_migrations", [])
                }
            )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"手动执行迁移失败: {e}")
        raise HTTPException(status_code=500, detail=f"迁移执行失败: {str(e)}")


@router.get("/schema/migrations", summary="获取所有迁移信息")
async def get_all_migrations(
    db: DatabaseManager = Depends(get_database)
):
    """获取所有迁移的详细信息"""
    try:
        from migrations.manager import migration_manager
        from migrations.loader import migration_loader

        # 获取迁移状态
        status = await migration_manager.get_migration_status()

        # 获取迁移详细信息
        migration_info = migration_loader.get_migration_info()

        return {
            "success": True,
            "data": {
                "status": status,
                "migrations": migration_info
            },
            "message": "获取迁移信息成功"
        }

    except Exception as e:
        logger.error(f"获取迁移信息失败: {e}")
        raise HTTPException(status_code=500, detail=f"获取迁移信息失败: {str(e)}")


@router.post("/schema/reload", summary="重新加载迁移文件")
async def reload_migrations(
    db: DatabaseManager = Depends(get_database)
):
    """重新加载迁移文件（开发环境使用）"""
    try:
        from migrations.loader import reload_migrations

        logger.info("开始重新加载迁移文件")
        migration_count = reload_migrations()

        return {
            "success": True,
            "data": {
                "loaded_migrations": migration_count
            },
            "message": f"成功重新加载 {migration_count} 个迁移文件"
        }

    except Exception as e:
        logger.error(f"重新加载迁移文件失败: {e}")
        raise HTTPException(status_code=500, detail=f"重新加载失败: {str(e)}")


@router.get("/schema/version", summary="获取当前数据库schema版本")
async def get_current_schema_version(
    db: DatabaseManager = Depends(get_database)
):
    """获取当前数据库schema版本"""
    try:
        from migrations.manager import migration_manager

        executed_migrations = await migration_manager.get_executed_migrations()

        # 获取最新执行的迁移版本
        latest_version = executed_migrations[-1] if executed_migrations else "0.0.0"

        return {
            "success": True,
            "data": {
                "current_version": latest_version,
                "executed_migrations": executed_migrations,
                "total_executed": len(executed_migrations)
            },
            "message": "获取当前版本成功"
        }

    except Exception as e:
        logger.error(f"获取当前版本失败: {e}")
        raise HTTPException(status_code=500, detail=f"获取版本失败: {str(e)}")
