"""
调度器配置管理 API
提供调度器配置的查看和修改功能
"""

from fastapi import APIRouter, HTTPException
from pydantic import BaseModel
from typing import Optional
from loguru import logger

from scheduler import monthly_scheduler
from models import BaseResponse

router = APIRouter()


class SchedulerConfig(BaseModel):
    """调度器配置模型"""
    auto_save_enabled: Optional[bool] = None
    auto_save_day: Optional[int] = None
    auto_save_hour: Optional[int] = None
    auto_save_minute: Optional[int] = None
    max_retries: Optional[int] = None


@router.get("/config", summary="获取调度器配置")
async def get_scheduler_config():
    """获取当前调度器配置"""
    try:
        config = monthly_scheduler.get_config()
        return BaseResponse(
            message="获取调度器配置成功",
            data=config
        )
    except Exception as e:
        logger.error(f"获取调度器配置失败: {e}")
        raise HTTPException(status_code=500, detail=f"获取配置失败: {str(e)}")


@router.put("/config", summary="更新调度器配置")
async def update_scheduler_config(config: SchedulerConfig):
    """更新调度器配置"""
    try:
        # 验证配置参数
        config_dict = config.dict(exclude_unset=True)
        
        # 验证参数范围
        if 'auto_save_day' in config_dict:
            if not (1 <= config_dict['auto_save_day'] <= 28):
                raise HTTPException(status_code=400, detail="自动保存日期必须在1-28之间")
                
        if 'auto_save_hour' in config_dict:
            if not (0 <= config_dict['auto_save_hour'] <= 23):
                raise HTTPException(status_code=400, detail="自动保存小时必须在0-23之间")
                
        if 'auto_save_minute' in config_dict:
            if not (0 <= config_dict['auto_save_minute'] <= 59):
                raise HTTPException(status_code=400, detail="自动保存分钟必须在0-59之间")
                
        if 'max_retries' in config_dict:
            if not (0 <= config_dict['max_retries'] <= 10):
                raise HTTPException(status_code=400, detail="最大重试次数必须在0-10之间")
        
        # 更新配置
        updated = monthly_scheduler.update_config(**config_dict)
        
        if not updated:
            return BaseResponse(message="没有配置需要更新")
            
        return BaseResponse(
            message=f"调度器配置更新成功: {', '.join(updated)}",
            data=monthly_scheduler.get_config()
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"更新调度器配置失败: {e}")
        raise HTTPException(status_code=500, detail=f"更新配置失败: {str(e)}")


@router.post("/trigger-save", summary="手动触发自动保存")
async def trigger_manual_save():
    """手动触发自动保存任务（用于测试）"""
    try:
        logger.info("🔧 收到手动触发自动保存请求")
        result = await monthly_scheduler.trigger_manual_save()
        
        if result:
            return BaseResponse(message="手动触发自动保存成功")
        else:
            return BaseResponse(message="手动触发自动保存失败", success=False)
            
    except Exception as e:
        logger.error(f"手动触发自动保存失败: {e}")
        raise HTTPException(status_code=500, detail=f"触发失败: {str(e)}")


@router.get("/status", summary="获取调度器状态")
async def get_scheduler_status():
    """获取调度器运行状态"""
    try:
        config = monthly_scheduler.get_config()
        
        # 获取下次执行时间
        next_run_time = None
        if monthly_scheduler.is_running and config['auto_save_enabled']:
            job = monthly_scheduler.scheduler.get_job('monthly_auto_save')
            if job:
                next_run_time = job.next_run_time.isoformat() if job.next_run_time else None
        
        status = {
            "is_running": config['is_running'],
            "auto_save_enabled": config['auto_save_enabled'],
            "next_run_time": next_run_time,
            "config": config
        }
        
        return BaseResponse(
            message="获取调度器状态成功",
            data=status
        )
        
    except Exception as e:
        logger.error(f"获取调度器状态失败: {e}")
        raise HTTPException(status_code=500, detail=f"获取状态失败: {str(e)}")
