"""
系统配置 API 路由
提供系统配置的 CRUD 操作接口
"""

from fastapi import APIRouter, HTTPException, Depends
from typing import List, Dict, Any
from loguru import logger

from database import system_config_table, get_database, DatabaseManager
from models import SystemConfigItem, SystemConfigResponse, BaseResponse, UserResponse
from dependencies import get_current_active_user, require_admin

router = APIRouter()


@router.get("", response_model=BaseResponse, summary="获取所有系统配置")
async def get_all_config(
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """获取所有系统配置，返回键值对格式"""
    try:
        query = system_config_table.select()
        configs = await db.fetch_all(query)

        # 转换为键值对格式，与前端localStorage格式保持一致
        result = {}
        for config in configs:
            result[config["config_key"]] = config["config_value"]

        return BaseResponse(
            success=True,
            message="获取系统配置成功",
            data=result
        )

    except Exception as e:
        logger.error(f"获取系统配置失败: {e}")
        raise HTTPException(status_code=500, detail="获取系统配置失败")


@router.get("/list", response_model=BaseResponse, summary="获取系统配置列表")
async def get_config_list(db: DatabaseManager = Depends(get_database)):
    """获取系统配置列表（详细格式）"""
    try:
        query = system_config_table.select().order_by(system_config_table.c.config_key)
        configs = await db.fetch_all(query)

        config_list = [
            SystemConfigResponse(
                id=config["id"],
                config_key=config["config_key"],
                config_value=config["config_value"],
                description=config["description"],
                created_at=config["created_at"],
                updated_at=config["updated_at"]
            )
            for config in configs
        ]

        return BaseResponse(
            success=True,
            message="获取系统配置列表成功",
            data=config_list
        )

    except Exception as e:
        logger.error(f"获取系统配置列表失败: {e}")
        raise HTTPException(status_code=500, detail="获取系统配置列表失败")


@router.get("/{config_key}", response_model=BaseResponse, summary="获取指定配置项")
async def get_config_by_key(config_key: str, db: DatabaseManager = Depends(get_database)):
    """根据配置键名获取配置值"""
    try:
        query = system_config_table.select().where(system_config_table.c.config_key == config_key)
        config = await db.fetch_one(query)

        if not config:
            raise HTTPException(status_code=404, detail="配置项不存在")

        return BaseResponse(
            success=True,
            message="获取配置项成功",
            data=config["config_value"]
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"获取配置项失败: {e}")
        raise HTTPException(status_code=500, detail="获取配置项失败")


@router.post("", response_model=BaseResponse, summary="创建或更新配置项")
async def set_config(
    config_item: SystemConfigItem,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """创建或更新系统配置项"""
    try:
        # 检查配置项是否已存在
        existing_config = await db.fetch_one(
            system_config_table.select().where(
                system_config_table.c.config_key == config_item.config_key
            )
        )
        
        if existing_config:
            # 更新现有配置
            query = system_config_table.update().where(
                system_config_table.c.config_key == config_item.config_key
            ).values(
                config_value=config_item.config_value,
                description=config_item.description
            )
            await db.execute(query)
            message = "配置项更新成功"
        else:
            # 创建新配置
            query = system_config_table.insert().values(
                config_key=config_item.config_key,
                config_value=config_item.config_value,
                description=config_item.description
            )
            await db.execute(query)
            message = "配置项创建成功"
        
        return BaseResponse(message=message)
        
    except Exception as e:
        logger.error(f"设置配置项失败: {e}")
        raise HTTPException(status_code=500, detail="设置配置项失败")


@router.put("/", response_model=BaseResponse, summary="批量更新配置")
async def update_multiple_configs(
    configs: Dict[str, Any],
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """批量更新系统配置"""
    try:
        async with db.transaction():
            for key, value in configs.items():
                # 检查配置项是否存在
                existing_config = await db.fetch_one(
                    system_config_table.select().where(
                        system_config_table.c.config_key == key
                    )
                )
                
                if existing_config:
                    # 更新现有配置
                    query = system_config_table.update().where(
                        system_config_table.c.config_key == key
                    ).values(config_value=value)
                    await db.execute(query)
                else:
                    # 创建新配置
                    query = system_config_table.insert().values(
                        config_key=key,
                        config_value=value,
                        description=f"系统配置项: {key}"
                    )
                    await db.execute(query)
        
        return BaseResponse(message=f"成功更新 {len(configs)} 个配置项")
        
    except Exception as e:
        logger.error(f"批量更新配置失败: {e}")
        raise HTTPException(status_code=500, detail="批量更新配置失败")


@router.delete("/{config_key}", response_model=BaseResponse, summary="删除配置项")
async def delete_config(
    config_key: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """删除系统配置项"""
    try:
        # 检查配置项是否存在
        existing_config = await db.fetch_one(
            system_config_table.select().where(system_config_table.c.config_key == config_key)
        )
        if not existing_config:
            raise HTTPException(status_code=404, detail="配置项不存在")
        
        # 不允许删除关键系统配置
        protected_keys = ["initialized", "version"]
        if config_key in protected_keys:
            raise HTTPException(status_code=400, detail="不能删除系统关键配置")
        
        # 删除配置项
        query = system_config_table.delete().where(system_config_table.c.config_key == config_key)
        await db.execute(query)
        
        return BaseResponse(message="配置项删除成功")
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"删除配置项失败: {e}")
        raise HTTPException(status_code=500, detail="删除配置项失败")


@router.post("/reset", response_model=BaseResponse, summary="重置为默认配置")
async def reset_to_default(
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """重置系统配置为默认值"""
    try:
        # 默认配置项
        default_configs = [
            ("initialized", True, "系统是否已初始化"),
            ("version", "8.8.0", "系统版本"),
            ("hasDefaultData", False, "是否有默认示例数据")
        ]
        
        async with db.transaction():
            # 清空现有配置
            await db.execute(system_config_table.delete())
            
            # 插入默认配置
            for key, value, desc in default_configs:
                query = system_config_table.insert().values(
                    config_key=key,
                    config_value=value,
                    description=desc
                )
                await db.execute(query)
        
        return BaseResponse(message="系统配置已重置为默认值")
        
    except Exception as e:
        logger.error(f"重置配置失败: {e}")
        raise HTTPException(status_code=500, detail="重置配置失败")
