"""
用户管理 API 路由
提供用户的 CRUD 操作接口
"""

from fastapi import APIRouter, HTTPException, Depends
from typing import List
from loguru import logger

from database import users_table, get_database, DatabaseManager
from models import (
    UserCreate, UserUpdate, UserResponse, BaseResponse,
    LoginRequest, LoginResponse, TokenRefreshRequest, TokenRefreshResponse
)
from auth import jwt_manager, password_manager
from dependencies import (
    get_current_user, get_current_active_user, require_admin,
    token_blacklist, verify_token_not_blacklisted
)
from config import settings

router = APIRouter()


@router.get("", response_model=BaseResponse, summary="获取所有用户")
async def get_all_users(
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """获取所有用户列表"""
    try:
        query = users_table.select()
        users = await db.fetch_all(query)

        user_list = [
            UserResponse(
                id=user["id"],
                phone=user["phone"],
                name=user["name"],
                role=user["role"],
                institutions=user["institutions"] or [],
                created_at=user["created_at"],
                updated_at=user["updated_at"]
            )
            for user in users
        ]

        return BaseResponse(
            success=True,
            message="获取用户列表成功",
            data=user_list
        )
    except Exception as e:
        logger.error(f"获取用户列表失败: {e}")
        raise HTTPException(status_code=500, detail="获取用户列表失败")


@router.get("/me", response_model=UserResponse, summary="获取当前用户信息")
async def get_current_user_info(current_user: UserResponse = Depends(get_current_active_user)):
    """获取当前登录用户的信息"""
    return current_user


@router.get("/{user_id}", response_model=BaseResponse, summary="根据ID获取用户")
async def get_user_by_id(
    user_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(get_current_active_user)
):
    """根据用户ID获取用户信息"""
    try:
        query = users_table.select().where(users_table.c.id == user_id)
        user = await db.fetch_one(query)

        if not user:
            raise HTTPException(status_code=404, detail="用户不存在")

        user_data = UserResponse(
            id=user["id"],
            phone=user["phone"],
            name=user["name"],
            role=user["role"],
            institutions=user["institutions"] or [],
            created_at=user["created_at"],
            updated_at=user["updated_at"]
        )

        return BaseResponse(
            success=True,
            message="获取用户信息成功",
            data=user_data
        )
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"获取用户失败: {e}")
        raise HTTPException(status_code=500, detail="获取用户失败")


@router.post("", response_model=BaseResponse, summary="创建用户")
async def create_user(
    user_data: UserCreate,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """创建新用户"""
    try:
        # 检查用户ID是否已存在
        existing_user = await db.fetch_one(
            users_table.select().where(users_table.c.id == user_data.id)
        )
        if existing_user:
            raise HTTPException(status_code=400, detail="用户ID已存在")

        # 检查手机号是否已存在
        existing_phone = await db.fetch_one(
            users_table.select().where(users_table.c.phone == user_data.phone)
        )
        if existing_phone:
            raise HTTPException(status_code=400, detail="手机号已存在")
        
        # 加密密码
        hashed_password = password_manager.hash_password(user_data.password)

        # 插入新用户
        query = users_table.insert().values(
            id=user_data.id,
            phone=user_data.phone,
            password=hashed_password,
            name=user_data.name,
            role=user_data.role,
            institutions=user_data.institutions
        )
        
        await db.execute(query)

        # 获取创建的用户信息
        created_user_response = await get_user_by_id(user_data.id, db, current_user)

        return BaseResponse(
            success=True,
            message="创建用户成功",
            data=created_user_response.data
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"创建用户失败: {e}")
        raise HTTPException(status_code=500, detail="创建用户失败")


@router.put("/{user_id}", response_model=BaseResponse, summary="更新用户")
async def update_user(
    user_id: str,
    user_data: UserUpdate,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """更新用户信息"""
    try:
        # 检查用户是否存在
        existing_user = await db.fetch_one(
            users_table.select().where(users_table.c.id == user_id)
        )
        if not existing_user:
            raise HTTPException(status_code=404, detail="用户不存在")

        # 构建更新数据
        update_data = {}
        if user_data.phone is not None:
            # 检查手机号是否被其他用户使用
            phone_check = await db.fetch_one(
                users_table.select().where(
                    (users_table.c.phone == user_data.phone) &
                    (users_table.c.id != user_id)
                )
            )
            if phone_check:
                raise HTTPException(status_code=400, detail="手机号已被其他用户使用")
            update_data["phone"] = user_data.phone

        if user_data.name is not None:
            update_data["name"] = user_data.name
        if user_data.password is not None:
            # 加密新密码
            update_data["password"] = password_manager.hash_password(user_data.password)
        if user_data.role is not None:
            update_data["role"] = user_data.role
        if user_data.institutions is not None:
            update_data["institutions"] = user_data.institutions

        if not update_data:
            raise HTTPException(status_code=400, detail="没有提供更新数据")

        # 执行更新
        query = users_table.update().where(users_table.c.id == user_id).values(**update_data)
        await db.execute(query)

        # 获取更新后的用户信息
        updated_user_response = await get_user_by_id(user_id, db, current_user)

        return BaseResponse(
            success=True,
            message="更新用户成功",
            data=updated_user_response.data
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"更新用户失败: {e}")
        raise HTTPException(status_code=500, detail="更新用户失败")


@router.delete("/{user_id}", response_model=BaseResponse, summary="删除用户")
async def delete_user(
    user_id: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """删除用户"""
    try:
        # 检查用户是否存在
        existing_user = await db.fetch_one(
            users_table.select().where(users_table.c.id == user_id)
        )
        if not existing_user:
            raise HTTPException(status_code=404, detail="用户不存在")
        
        # 不允许删除管理员用户
        if existing_user["role"] == "admin":
            raise HTTPException(status_code=400, detail="不能删除管理员用户")
        
        # 删除用户（外键约束会自动处理相关数据）
        query = users_table.delete().where(users_table.c.id == user_id)
        await db.execute(query)
        
        return BaseResponse(message="用户删除成功")
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"删除用户失败: {e}")
        raise HTTPException(status_code=500, detail="删除用户失败")


@router.post("/login", response_model=LoginResponse, summary="用户登录")
async def login(login_data: LoginRequest, db: DatabaseManager = Depends(get_database)):
    """用户登录验证"""
    try:
        # 查找用户
        query = users_table.select().where(users_table.c.phone == login_data.phone)
        user = await db.fetch_one(query)

        if not user:
            logger.warning(f"登录失败：用户不存在 - {login_data.phone}")
            raise HTTPException(status_code=401, detail="手机号或密码错误")

        # 验证密码（支持明文和加密密码）
        password_valid = False
        if password_manager.is_hashed_password(user["password"]):
            # 已加密的密码，使用bcrypt验证
            password_valid = password_manager.verify_password(login_data.password, user["password"])
        else:
            # 明文密码，直接比较（兼容旧数据）
            password_valid = user["password"] == login_data.password

            # 如果明文密码验证成功，自动升级为加密密码
            if password_valid:
                hashed_password = password_manager.hash_password(login_data.password)
                update_query = users_table.update().where(
                    users_table.c.id == user["id"]
                ).values(password=hashed_password)
                await db.execute(update_query)
                logger.info(f"用户 {user['name']} 的密码已自动升级为加密存储")

        if not password_valid:
            logger.warning(f"登录失败：密码错误 - {login_data.phone}")
            raise HTTPException(status_code=401, detail="手机号或密码错误")

        # 创建用户信息
        user_response = UserResponse(
            id=user["id"],
            phone=user["phone"],
            name=user["name"],
            role=user["role"],
            institutions=user["institutions"] or [],
            created_at=user["created_at"],
            updated_at=user["updated_at"]
        )

        # 创建JWT token数据
        token_data = {
            "sub": user["phone"],  # subject (用户标识)
            "user_id": user["id"],
            "role": user["role"],
            "phone": user["phone"]
        }

        # 生成tokens
        access_token = jwt_manager.create_access_token(token_data)
        refresh_token = jwt_manager.create_refresh_token(token_data)

        logger.info(f"用户登录成功: {user['name']} ({user['phone']})")

        return LoginResponse(
            success=True,
            access_token=access_token,
            refresh_token=refresh_token,
            token_type="bearer",
            expires_in=settings.JWT_EXPIRE_HOURS * 3600,  # 转换为秒
            user=user_response,
            message="登录成功"
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"用户登录失败: {e}")
        raise HTTPException(status_code=500, detail="登录失败")


@router.post("/refresh", response_model=TokenRefreshResponse, summary="刷新访问token")
async def refresh_token(refresh_data: TokenRefreshRequest):
    """使用刷新token获取新的访问token"""
    try:
        # 生成新的访问token
        new_access_token = jwt_manager.refresh_access_token(refresh_data.refresh_token)

        logger.info("Token刷新成功")

        return TokenRefreshResponse(
            success=True,
            access_token=new_access_token,
            token_type="bearer",
            expires_in=settings.JWT_EXPIRE_HOURS * 3600,
            message="Token刷新成功"
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Token刷新失败: {e}")
        raise HTTPException(status_code=500, detail="Token刷新失败")


@router.post("/logout", response_model=BaseResponse, summary="用户登出")
async def logout(
    current_user: UserResponse = Depends(get_current_active_user),
    credentials = Depends(verify_token_not_blacklisted)
):
    """用户登出，将token加入黑名单"""
    try:
        # 将当前token加入黑名单
        token_blacklist.add_token(credentials.credentials)

        logger.info(f"用户登出成功: {current_user.name}")

        return BaseResponse(
            success=True,
            message="登出成功"
        )

    except Exception as e:
        logger.error(f"用户登出失败: {e}")
        raise HTTPException(status_code=500, detail="登出失败")



@router.get("/phone/{phone}", response_model=BaseResponse, summary="根据手机号获取用户")
async def get_user_by_phone(
    phone: str,
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """根据手机号获取用户信息"""
    try:
        query = users_table.select().where(users_table.c.phone == phone)
        user = await db.fetch_one(query)

        if not user:
            raise HTTPException(status_code=404, detail="用户不存在")

        user_data = UserResponse(
            id=user["id"],
            phone=user["phone"],
            name=user["name"],
            role=user["role"],
            institutions=user["institutions"] or [],
            created_at=user["created_at"],
            updated_at=user["updated_at"]
        )

        return BaseResponse(
            success=True,
            message="获取用户信息成功",
            data=user_data
        )
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"根据手机号获取用户失败: {e}")
        raise HTTPException(status_code=500, detail="获取用户失败")


@router.post("/migrate-passwords", response_model=BaseResponse, summary="迁移明文密码为加密密码")
async def migrate_passwords(
    db: DatabaseManager = Depends(get_database),
    current_user: UserResponse = Depends(require_admin)
):
    """将数据库中的明文密码迁移为加密密码"""
    try:
        # 获取所有用户
        query = users_table.select()
        users = await db.fetch_all(query)

        migrated_count = 0
        for user in users:
            # 检查密码是否已经是哈希格式
            if not password_manager.is_hashed_password(user["password"]):
                # 加密明文密码
                hashed_password = password_manager.hash_password(user["password"])

                # 更新数据库
                update_query = users_table.update().where(
                    users_table.c.id == user["id"]
                ).values(password=hashed_password)
                await db.execute(update_query)

                migrated_count += 1
                logger.info(f"用户 {user['name']} 的密码已迁移为加密存储")

        return BaseResponse(
            success=True,
            message=f"密码迁移完成，共迁移 {migrated_count} 个用户的密码"
        )

    except Exception as e:
        logger.error(f"密码迁移失败: {e}")
        raise HTTPException(status_code=500, detail="密码迁移失败")
