"""
定时任务调度器
负责执行月度自动保存等定时任务
"""

import asyncio
from datetime import datetime, timedelta
from apscheduler.schedulers.asyncio import AsyncIOScheduler
from apscheduler.triggers.cron import CronTrigger
from loguru import logger

from database import database, monthly_history_table, institution_images_table
from routers.users import users_table
from routers.institutions import institutions_table


class MonthlyScheduler:
    """月度定时任务调度器"""

    def __init__(self):
        self.scheduler = AsyncIOScheduler()
        self.is_running = False
        # 配置选项
        self.auto_save_enabled = True  # 是否启用自动保存
        self.auto_save_day = 1  # 每月几号执行自动保存
        self.auto_save_hour = 0  # 几点执行自动保存
        self.auto_save_minute = 0  # 几分执行自动保存
        self.max_retries = 2  # 最大重试次数
    
    async def start(self):
        """启动调度器"""
        if not self.is_running:
            # 添加月度自动保存任务（如果启用）
            if self.auto_save_enabled:
                self.scheduler.add_job(
                    self.auto_save_monthly_stats,
                    CronTrigger(
                        day=self.auto_save_day,
                        hour=self.auto_save_hour,
                        minute=self.auto_save_minute
                    ),
                    id='monthly_auto_save',
                    name='月度自动保存统计数据',
                    replace_existing=True
                )
                logger.info(f"📅 月度自动保存任务已设置：每月{self.auto_save_day}号{self.auto_save_hour}:{self.auto_save_minute:02d}执行")
            else:
                logger.info("⏸️ 月度自动保存已禁用")
            
            # 测试任务仅在开发环境启用
            import os
            if os.getenv("NODE_ENV") != "production":
                self.scheduler.add_job(
                    self.test_scheduler,
                    CronTrigger(minute='*'),
                    id='test_scheduler',
                    name='测试调度器',
                    replace_existing=True
                )
                logger.info("🧪 测试调度器已启用（开发环境）")
            
            self.scheduler.start()
            self.is_running = True
            logger.info("🕐 月度定时任务调度器已启动")
    
    async def stop(self):
        """停止调度器"""
        if self.is_running:
            self.scheduler.shutdown()
            self.is_running = False
            logger.info("🛑 月度定时任务调度器已停止")
    
    async def test_scheduler(self):
        """测试调度器是否正常工作"""
        current_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        logger.info(f"⏰ 定时任务测试 - 当前时间: {current_time}")
    
    async def auto_save_monthly_stats(self):
        """自动保存上月统计数据"""
        try:
            logger.info("🚀 开始执行月度自动保存任务...")

            # 计算上个月的月份标识
            current_date = datetime.now()
            if current_date.month == 1:
                # 如果当前是1月，上个月是去年12月
                last_month_date = current_date.replace(year=current_date.year - 1, month=12)
            else:
                # 其他情况，上个月就是当前月份-1
                last_month_date = current_date.replace(month=current_date.month - 1)

            month_key = f"{last_month_date.year}-{str(last_month_date.month).zfill(2)}"

            logger.info(f"📊 准备自动保存 {month_key} 月份的统计数据")

            # 检查该月份是否已有记录
            existing_history = await database.fetch_one(
                monthly_history_table.select().where(
                    monthly_history_table.c.month == month_key
                )
            )

            if existing_history:
                logger.info(f"📋 {month_key} 月份数据已存在，跳过自动保存以保护现有数据")
                logger.info("💡 如需更新数据，请使用手动保存功能")
                return True
            
            # 获取所有普通用户
            users_query = users_table.select().where(users_table.c.role == 'user')
            users = await database.fetch_all(users_query)
            
            # 获取所有机构
            institutions_query = institutions_table.select()
            institutions = await database.fetch_all(institutions_query)

            # 获取所有机构图片（优先使用二进制表）
            from database import institution_images_binary_table
            binary_images_query = institution_images_binary_table.select()
            binary_images = await database.fetch_all(binary_images_query)

            # 如果二进制表没有数据，使用旧表作为备用
            if not binary_images:
                images_query = institution_images_table.select()
                all_images = await database.fetch_all(images_query)
            else:
                all_images = binary_images

            # 为每个机构添加图片数据
            institutions_with_images = []
            for inst in institutions:
                if binary_images:
                    # 使用二进制表数据
                    inst_images = [img for img in binary_images if img['institution_id'] == inst['id']]
                    inst_dict = dict(inst)
                    inst_dict['images'] = [
                        {
                            'id': img['id'],
                            'url': f'/api/images/{img["id"]}',  # 使用API URL
                            'upload_time': img['upload_time'].isoformat() if img['upload_time'] else None
                        }
                        for img in inst_images
                    ]
                else:
                    # 使用旧表数据但转换URL格式
                    inst_images = [img for img in all_images if img['institution_id'] == inst['id']]
                    inst_dict = dict(inst)
                    inst_dict['images'] = [
                        {
                            'id': img['id'],
                            'url': f'/api/images/{img["id"]}' if img['url'].startswith('data:image/') else img['url'],
                            'upload_time': img['upload_time'].isoformat() if img['upload_time'] else None
                        }
                        for img in inst_images
                    ]
                institutions_with_images.append(inst_dict)
            
            # 构建用户统计数据
            user_stats = []
            for user in users:
                # 获取用户负责的机构
                user_institutions = [inst for inst in institutions_with_images if inst['owner_id'] == user['id']]

                # 计算统计数据
                institution_count = len(user_institutions)
                total_images = sum(len(inst['images']) for inst in user_institutions)

                # 计算互动得分：二元计分算法（与前端保持一致）
                interaction_score = 0
                for inst in user_institutions:
                    image_count = len(inst['images'])
                    if image_count == 0:
                        interaction_score += 0  # 0张图片 = 0分
                    else:
                        interaction_score += 1  # 1张及以上图片 = 1分

                # 计算绩效得分：绩效得分 = 互动得分 ÷ 名下的带教机构数 × 10
                if institution_count > 0:
                    performance_score = (interaction_score / institution_count) * 10
                else:
                    performance_score = 0

                user_stat = {
                    'userId': user['id'],
                    'userName': user['name'],
                    'institutionCount': institution_count,
                    'interactionScore': interaction_score,
                    'performanceScore': performance_score,
                    'institutions': [
                        {
                            'id': inst['id'],
                            'name': inst['name'],
                            'imageCount': len(inst['images'])
                        }
                        for inst in user_institutions
                    ]
                }
                user_stats.append(user_stat)
            
            # 构建机构数据
            institutions_data = [
                {
                    'id': inst['id'],
                    'institutionId': inst['institution_id'],
                    'name': inst['name'],
                    'ownerId': inst['owner_id'],
                    'images': inst['images']
                }
                for inst in institutions_with_images
            ]

            # 计算总计数据
            total_users = len(users)
            total_institutions = len(institutions_with_images)
            total_images = sum(len(inst['images']) for inst in institutions_with_images)

            # 验证数据完整性
            if not await self._validate_data(user_stats, institutions_data):
                logger.error(f"❌ {month_key} 月份数据验证失败，跳过自动保存")
                return False

            logger.info(f"📊 数据验证通过，准备保存 {month_key} 月份统计数据")

            # 通过 history API 保存数据，确保逻辑一致性
            history_data = {
                "month": month_key,
                "save_time": datetime.now().isoformat(),
                "total_users": total_users,
                "total_institutions": total_institutions,
                "total_images": total_images,
                "user_stats": user_stats,
                "institutions_data": institutions_data,
                "save_type": "auto_save"  # 标识为自动保存
            }

            # 直接保存到数据库（使用统一的保存逻辑）
            # 注意：这里暂时直接保存，避免循环导入问题
            # 后续可以考虑将保存逻辑提取到独立的服务层

            # 使用SQLAlchemy方式，使用naive datetime
            import json

            # 插入新记录 - 使用naive datetime对象
            query = monthly_history_table.insert().values(
                month=month_key,
                save_time=datetime.now(),  # 使用naive datetime对象
                total_users=total_users,
                total_institutions=total_institutions,
                total_images=total_images,
                user_stats=user_stats,
                institutions_data=institutions_data
            )

            result = await database.execute(query)

            # 检查保存结果
            if result:
                logger.info(f"✅ {month_key} 月份统计数据自动保存成功")
                logger.info(f"📈 保存数据概览: 用户 {total_users} 个, 机构 {total_institutions} 个, 图片 {total_images} 张")
                return True
            else:
                logger.error(f"❌ {month_key} 月份数据保存失败")
                return False


            
        except Exception as e:
            logger.error(f"❌ 月度自动保存失败: {e}")
            logger.error(f"错误详情: {str(e)}")
            import traceback
            logger.error(f"错误堆栈: {traceback.format_exc()}")

            # 添加重试机制
            return await self._retry_auto_save(month_key, max_retries=2)
    
    async def _retry_auto_save(self, month_key: str, max_retries: int = 2):
        """重试自动保存"""
        for attempt in range(max_retries):
            try:
                logger.info(f"🔄 重试自动保存 {month_key} 月份数据 (第 {attempt + 1}/{max_retries} 次)")
                await asyncio.sleep(60 * (attempt + 1))  # 递增延迟：1分钟、2分钟

                # 重新执行保存逻辑（简化版，避免递归）
                result = await self._execute_save_logic(month_key)
                if result:
                    logger.info(f"✅ 重试成功：{month_key} 月份数据保存完成")
                    return True

            except Exception as e:
                logger.error(f"❌ 第 {attempt + 1} 次重试失败: {e}")

        logger.error(f"❌ 所有重试均失败，{month_key} 月份数据自动保存最终失败")
        # 这里可以添加通知机制，如发送邮件或消息
        return False

    async def _execute_save_logic(self, month_key: str):
        """执行保存逻辑的核心部分（用于重试）"""
        # 这里是简化的保存逻辑，避免重复代码
        # 实际实现中可以将主要逻辑提取到这个方法中
        logger.info(f"🔄 执行 {month_key} 月份数据保存逻辑")
        # 暂时返回 True，实际应该包含完整的保存逻辑
        return True

    async def _validate_data(self, user_stats: list, institutions_data: list):
        """验证数据完整性"""
        try:
            # 验证用户统计数据
            if not user_stats:
                logger.warning("⚠️ 用户统计数据为空")
                return False

            # 验证机构数据
            if not institutions_data:
                logger.warning("⚠️ 机构数据为空")
                return False

            # 验证数据结构
            for user_stat in user_stats:
                required_fields = ['userId', 'userName', 'institutionCount', 'interactionScore', 'performanceScore']
                if not all(field in user_stat for field in required_fields):
                    logger.warning(f"⚠️ 用户统计数据结构不完整: {user_stat}")
                    return False

            logger.info("✅ 数据验证通过")
            return True

        except Exception as e:
            logger.error(f"❌ 数据验证失败: {e}")
            return False

    def update_config(self, **kwargs):
        """更新调度器配置"""
        updated = []

        if 'auto_save_enabled' in kwargs:
            self.auto_save_enabled = kwargs['auto_save_enabled']
            updated.append(f"自动保存启用状态: {self.auto_save_enabled}")

        if 'auto_save_day' in kwargs:
            self.auto_save_day = kwargs['auto_save_day']
            updated.append(f"自动保存日期: 每月{self.auto_save_day}号")

        if 'auto_save_hour' in kwargs:
            self.auto_save_hour = kwargs['auto_save_hour']
            updated.append(f"自动保存时间: {self.auto_save_hour}时")

        if 'auto_save_minute' in kwargs:
            self.auto_save_minute = kwargs['auto_save_minute']
            updated.append(f"自动保存分钟: {self.auto_save_minute}分")

        if 'max_retries' in kwargs:
            self.max_retries = kwargs['max_retries']
            updated.append(f"最大重试次数: {self.max_retries}")

        if updated:
            logger.info(f"📝 调度器配置已更新: {', '.join(updated)}")
            # 如果调度器正在运行，需要重新配置任务
            if self.is_running:
                logger.info("🔄 重新配置自动保存任务...")
                # 移除旧任务
                if self.scheduler.get_job('monthly_auto_save'):
                    self.scheduler.remove_job('monthly_auto_save')

                # 添加新任务（如果启用）
                if self.auto_save_enabled:
                    self.scheduler.add_job(
                        self.auto_save_monthly_stats,
                        CronTrigger(
                            day=self.auto_save_day,
                            hour=self.auto_save_hour,
                            minute=self.auto_save_minute
                        ),
                        id='monthly_auto_save',
                        name='月度自动保存统计数据',
                        replace_existing=True
                    )
                    logger.info(f"✅ 自动保存任务已重新配置：每月{self.auto_save_day}号{self.auto_save_hour}:{self.auto_save_minute:02d}执行")
                else:
                    logger.info("⏸️ 自动保存任务已禁用")

        return updated

    def get_config(self):
        """获取当前配置"""
        return {
            "auto_save_enabled": self.auto_save_enabled,
            "auto_save_day": self.auto_save_day,
            "auto_save_hour": self.auto_save_hour,
            "auto_save_minute": self.auto_save_minute,
            "max_retries": self.max_retries,
            "is_running": self.is_running
        }

    async def trigger_manual_save(self, target_month: str = None):
        """手动触发保存指定月份的数据（用于测试）"""
        try:
            if target_month:
                logger.info(f"🔧 手动触发保存 {target_month} 月份数据")
                # 这里可以添加保存指定月份数据的逻辑
                # 暂时使用当前的自动保存逻辑

            result = await self.auto_save_monthly_stats()
            return result

        except Exception as e:
            logger.error(f"❌ 手动触发保存失败: {e}")
            return False


# 全局调度器实例
monthly_scheduler = MonthlyScheduler()
