-- 绩效计分系统数据库初始化脚本
-- 基于localStorage数据结构设计的PostgreSQL表结构

-- 删除已存在的表（开发环境）
DROP TABLE IF EXISTS monthly_history CASCADE;
DROP TABLE IF EXISTS institution_images CASCADE;
DROP TABLE IF EXISTS institutions CASCADE;
DROP TABLE IF EXISTS users CASCADE;
DROP TABLE IF EXISTS system_config CASCADE;

-- 创建用户表 (对应 localStorage: score_system_users)
CREATE TABLE users (
    id VARCHAR(50) PRIMARY KEY,                    -- 用户ID，如 'admin', 'user_1234567890'
    phone VARCHAR(20) UNIQUE NOT NULL,             -- 手机号，用于登录
    password VARCHAR(255) NOT NULL,                -- 密码（明文存储，与原系统保持一致）
    name VARCHAR(100) NOT NULL,                    -- 用户姓名
    role VARCHAR(20) DEFAULT 'user',               -- 角色：'admin' 或 'user'
    institutions JSONB DEFAULT '[]',               -- 负责的机构ID数组，保持原有结构
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 创建机构表 (对应 localStorage: score_system_institutions)
CREATE TABLE institutions (
    id VARCHAR(50) PRIMARY KEY,                    -- 内部ID，如 'inst_1234567890_abc123'
    institution_id VARCHAR(50),                   -- 机构编号，如 '001', '002'
    name VARCHAR(200) NOT NULL,                   -- 机构名称
    owner_id VARCHAR(50),                         -- 负责人ID，可为NULL（公池机构）
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- 外键约束
    CONSTRAINT fk_institutions_owner FOREIGN KEY (owner_id) REFERENCES users(id) ON DELETE SET NULL,
    
    -- 唯一约束
    CONSTRAINT uk_institutions_institution_id UNIQUE (institution_id)
);

-- 创建机构图片表 (对应 institutions.images 数组)
CREATE TABLE institution_images (
    id VARCHAR(50) PRIMARY KEY,                   -- 图片ID
    institution_id VARCHAR(50) NOT NULL,         -- 所属机构ID
    url TEXT NOT NULL,                           -- 图片URL/Base64数据
    upload_time TIMESTAMP NOT NULL,             -- 上传时间
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- 外键约束
    CONSTRAINT fk_images_institution FOREIGN KEY (institution_id) REFERENCES institutions(id) ON DELETE CASCADE
);

-- 创建系统配置表 (对应 localStorage: score_system_config)
CREATE TABLE system_config (
    id SERIAL PRIMARY KEY,
    config_key VARCHAR(100) UNIQUE NOT NULL,     -- 配置键名
    config_value JSONB,                          -- 配置值（JSON格式）
    description TEXT,                            -- 配置描述
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 创建月度历史统计表 (对应 localStorage: score_system_history)
CREATE TABLE monthly_history (
    id SERIAL PRIMARY KEY,
    month VARCHAR(7) NOT NULL,                   -- 月份，格式：YYYY-MM
    save_time TIMESTAMP NOT NULL,               -- 保存时间
    total_users INTEGER NOT NULL,               -- 总用户数
    total_institutions INTEGER NOT NULL,        -- 总机构数
    total_images INTEGER NOT NULL,              -- 总图片数
    user_stats JSONB NOT NULL,                  -- 用户统计数据（JSON格式）
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- 唯一约束：每月只能有一条记录
    CONSTRAINT uk_monthly_history_month UNIQUE (month)
);

-- 创建索引以提高查询性能
CREATE INDEX idx_users_phone ON users(phone);
CREATE INDEX idx_users_role ON users(role);
CREATE INDEX idx_institutions_owner_id ON institutions(owner_id);
CREATE INDEX idx_institutions_institution_id ON institutions(institution_id);
CREATE INDEX idx_institution_images_institution_id ON institution_images(institution_id);
CREATE INDEX idx_institution_images_upload_time ON institution_images(upload_time);
CREATE INDEX idx_system_config_key ON system_config(config_key);
CREATE INDEX idx_monthly_history_month ON monthly_history(month);

-- 创建更新时间触发器函数
CREATE OR REPLACE FUNCTION update_updated_at_column()
RETURNS TRIGGER AS $$
BEGIN
    NEW.updated_at = CURRENT_TIMESTAMP;
    RETURN NEW;
END;
$$ language 'plpgsql';

-- 为相关表创建更新时间触发器
CREATE TRIGGER update_users_updated_at 
    BEFORE UPDATE ON users
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_institutions_updated_at 
    BEFORE UPDATE ON institutions
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_system_config_updated_at 
    BEFORE UPDATE ON system_config
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

-- 插入默认系统配置
INSERT INTO system_config (config_key, config_value, description) VALUES
('initialized', 'true', '系统是否已初始化'),
('version', '"8.8.0"', '系统版本'),
('hasDefaultData', 'false', '是否有默认示例数据')
ON CONFLICT (config_key) DO UPDATE SET 
    config_value = EXCLUDED.config_value,
    updated_at = CURRENT_TIMESTAMP;

-- 插入默认管理员用户
INSERT INTO users (id, phone, password, name, role, institutions) VALUES
('admin', 'admin', 'admin123', '系统管理员', 'admin', '[]')
ON CONFLICT (id) DO UPDATE SET 
    phone = EXCLUDED.phone,
    password = EXCLUDED.password,
    name = EXCLUDED.name,
    role = EXCLUDED.role,
    institutions = EXCLUDED.institutions,
    updated_at = CURRENT_TIMESTAMP;

-- 创建数据迁移辅助视图
CREATE OR REPLACE VIEW v_institutions_with_images AS
SELECT 
    i.id,
    i.institution_id,
    i.name,
    i.owner_id,
    i.created_at,
    i.updated_at,
    COALESCE(
        JSON_AGG(
            JSON_BUILD_OBJECT(
                'id', img.id,
                'url', img.url,
                'uploadTime', img.upload_time
            ) ORDER BY img.upload_time
        ) FILTER (WHERE img.id IS NOT NULL),
        '[]'::json
    ) AS images
FROM institutions i
LEFT JOIN institution_images img ON i.id = img.institution_id
GROUP BY i.id, i.institution_id, i.name, i.owner_id, i.created_at, i.updated_at;

-- 创建数据完整性检查函数
CREATE OR REPLACE FUNCTION check_data_integrity()
RETURNS TABLE(
    check_name TEXT,
    status TEXT,
    message TEXT,
    count INTEGER
) AS $$
BEGIN
    -- 检查孤立机构（负责人不存在）
    RETURN QUERY
    SELECT 
        'orphan_institutions'::TEXT,
        CASE WHEN COUNT(*) = 0 THEN 'OK' ELSE 'WARNING' END::TEXT,
        ('发现 ' || COUNT(*) || ' 个孤立机构')::TEXT,
        COUNT(*)::INTEGER
    FROM institutions i
    LEFT JOIN users u ON i.owner_id = u.id
    WHERE i.owner_id IS NOT NULL AND u.id IS NULL;
    
    -- 检查重复的机构编号
    RETURN QUERY
    SELECT 
        'duplicate_institution_ids'::TEXT,
        CASE WHEN COUNT(*) = 0 THEN 'OK' ELSE 'ERROR' END::TEXT,
        ('发现 ' || COUNT(*) || ' 个重复的机构编号')::TEXT,
        COUNT(*)::INTEGER
    FROM (
        SELECT institution_id, COUNT(*) as cnt
        FROM institutions 
        WHERE institution_id IS NOT NULL
        GROUP BY institution_id
        HAVING COUNT(*) > 1
    ) duplicates;
    
    -- 检查无效图片记录
    RETURN QUERY
    SELECT 
        'invalid_images'::TEXT,
        CASE WHEN COUNT(*) = 0 THEN 'OK' ELSE 'WARNING' END::TEXT,
        ('发现 ' || COUNT(*) || ' 个无效图片记录')::TEXT,
        COUNT(*)::INTEGER
    FROM institution_images img
    LEFT JOIN institutions i ON img.institution_id = i.id
    WHERE i.id IS NULL;
END;
$$ LANGUAGE plpgsql;

COMMENT ON TABLE users IS '用户表，存储系统用户信息';
COMMENT ON TABLE institutions IS '机构表，存储机构基本信息';
COMMENT ON TABLE institution_images IS '机构图片表，存储机构上传的图片信息';
COMMENT ON TABLE system_config IS '系统配置表，存储系统配置参数';
COMMENT ON TABLE monthly_history IS '月度历史统计表，存储每月的统计数据';
COMMENT ON VIEW v_institutions_with_images IS '机构及其图片的聚合视图，便于数据查询';
COMMENT ON FUNCTION check_data_integrity() IS '数据完整性检查函数，用于验证数据一致性';
