import { createRouter, createWebHistory } from 'vue-router'
import { useAuthStore } from '@/store/auth'

/**
 * 路由配置
 * 包含登录页面、用户操作页面、管理员面板页面的路由定义
 */
const routes = [
  {
    path: '/',
    redirect: '/login'
  },
  {
    path: '/login',
    name: 'Login',
    component: () => import('@/views/auth/Login.vue'),
    meta: {
      title: '登录',
      requiresAuth: false
    }
  },
  {
    path: '/user',
    name: 'User',
    component: () => import('@/views/user/UserPanel.vue'),
    meta: {
      title: '用户操作面板',
      requiresAuth: true,
      roles: ['user', 'admin']
    }
  },
  {
    path: '/admin',
    name: 'Admin',
    component: () => import('@/views/admin/AdminPanel.vue'),
    meta: {
      title: '管理员控制面板',
      requiresAuth: true,
      roles: ['admin']
    }
  }
]

const router = createRouter({
  history: createWebHistory(),
  routes
})

/**
 * 路由守卫 - 检查用户认证状态和权限
 */
router.beforeEach(async (to, from, next) => {
  const authStore = useAuthStore()

  // 设置页面标题
  if (to.meta.title) {
    document.title = `${to.meta.title} - 绩效计分系统`
  }

  // 如果是首次访问或刷新页面，且有token，等待认证状态恢复
  if (!authStore.isAuthenticated && localStorage.getItem('auth_tokens')) {
    try {
      await authStore.restoreAuth()
    } catch (error) {
      console.warn('认证状态恢复失败:', error)
    }
  }

  // 检查是否需要认证
  if (to.meta.requiresAuth) {
    if (!authStore.isAuthenticated) {
      // 未登录，跳转到登录页
      next('/login')
      return
    }

    // 检查角色权限 - 使用 effectiveUser 而不是 currentUser
    // 这样管理员切换到用户视图时，会使用 viewAsUser 的角色
    const effectiveRole = authStore.effectiveUser?.role || authStore.currentUser?.role
    if (to.meta.roles && !to.meta.roles.includes(effectiveRole)) {
      // 权限不足，跳转到用户面板
      next('/user')
      return
    }
  }

  // 已登录用户访问登录页，直接跳转到对应面板
  if (to.path === '/login' && authStore.isAuthenticated) {
    // 如果正在查看用户视图，跳转到用户面板
    if (authStore.isViewingAsUser) {
      next('/user')
    } else if (authStore.currentUser.role === 'admin') {
      next('/admin')
    } else {
      next('/user')
    }
    return
  }

  next()
})

export default router 