/**
 * API 服务封装层
 * 提供与后端 FastAPI 服务的通信接口
 * 替换原有的 localStorage 存储机制
 */

// API 基础配置
// 生产环境：VITE_API_BASE_URL 为空字符串，使用相对路径通过Nginx代理
// 开发环境：VITE_API_BASE_URL 为 http://localhost:8000，直接访问后端
const API_BASE_URL = import.meta.env.VITE_API_BASE_URL !== undefined
  ? import.meta.env.VITE_API_BASE_URL
  : 'http://localhost:8000'

/**
 * HTTP 请求封装类
 */
class ApiClient {
  constructor(baseURL = API_BASE_URL) {
    this.baseURL = baseURL
    this.timeout = 10000 // 10秒超时
  }

  /**
   * 获取认证头
   */
  getAuthHeaders() {
    const token = this.getAccessToken()
    if (token) {
      return {
        'Authorization': `Bearer ${token}`
      }
    }
    return {}
  }

  /**
   * 获取访问token
   */
  getAccessToken() {
    try {
      const authData = localStorage.getItem('auth_tokens')
      if (authData) {
        const tokens = JSON.parse(authData)
        return tokens.access_token
      }
    } catch (error) {
      console.error('获取访问token失败:', error)
    }
    return null
  }

  /**
   * 获取刷新token
   */
  getRefreshToken() {
    try {
      const authData = localStorage.getItem('auth_tokens')
      if (authData) {
        const tokens = JSON.parse(authData)
        return tokens.refresh_token
      }
    } catch (error) {
      console.error('获取刷新token失败:', error)
    }
    return null
  }

  /**
   * 保存tokens
   */
  saveTokens(accessToken, refreshToken) {
    try {
      const tokens = {
        access_token: accessToken,
        refresh_token: refreshToken,
        timestamp: Date.now()
      }
      localStorage.setItem('auth_tokens', JSON.stringify(tokens))
    } catch (error) {
      console.error('保存tokens失败:', error)
    }
  }

  /**
   * 清除tokens
   */
  clearTokens() {
    localStorage.removeItem('auth_tokens')
  }

  /**
   * 通用请求方法
   */
  async request(endpoint, options = {}) {
    const url = `${this.baseURL}${endpoint}`

    // 检查是否是FormData，如果是则不设置Content-Type
    const isFormData = options.body instanceof FormData

    const config = {
      headers: {
        // 只有在不是FormData时才设置JSON Content-Type
        ...(isFormData ? {} : { 'Content-Type': 'application/json' }),
        // 先设置传入的headers，再设置认证头，确保认证头不被覆盖
        ...options.headers,
        ...this.getAuthHeaders()
      },
      ...options
    }



    try {
      const controller = new AbortController()
      const timeoutId = setTimeout(() => controller.abort(), this.timeout)

      const response = await fetch(url, {
        ...config,
        signal: controller.signal
      })
      
      clearTimeout(timeoutId)

      // 处理401未授权响应
      if (response.status === 401) {
        // 尝试刷新token
        const refreshed = await this.refreshAccessToken()
        if (refreshed) {
          // 重新发送原始请求
          const retryConfig = {
            ...config,
            headers: {
              ...config.headers,
              ...this.getAuthHeaders()
            }
          }

          const retryResponse = await fetch(url, {
            ...retryConfig,
            signal: controller.signal
          })

          if (retryResponse.ok) {
            const retryData = await retryResponse.json()
            return this.unwrapResponse(retryData)
          }
        }

        // 刷新失败或重试失败，清除tokens并跳转到登录页
        this.clearTokens()
        this.handleAuthError()
        throw new Error('认证失败，请重新登录')
      }

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}))
        throw new Error(errorData.detail || `HTTP ${response.status}: ${response.statusText}`)
      }

      const responseData = await response.json()
      // 自动解包BaseResponse格式的响应
      return this.unwrapResponse(responseData)
    } catch (error) {
      if (error.name === 'AbortError') {
        throw new Error('请求超时，请检查网络连接')
      }
      console.error(`API请求失败 [${endpoint}]:`, error)
      throw error
    }
  }

  /**
   * 自动解包BaseResponse格式的响应
   */
  unwrapResponse(responseData) {
    // 如果响应包含success字段，说明是BaseResponse格式
    if (responseData && typeof responseData === 'object' && 'success' in responseData) {
      // 检查请求是否成功
      if (!responseData.success) {
        throw new Error(responseData.message || '请求失败')
      }
      // 返回data字段的内容，如果没有data字段则返回整个响应
      return responseData.data !== undefined ? responseData.data : responseData
    }
    // 如果不是BaseResponse格式，直接返回原始数据（如login/logout等特殊接口）
    return responseData
  }

  /**
   * 刷新访问token
   */
  async refreshAccessToken() {
    try {
      const refreshToken = this.getRefreshToken()
      if (!refreshToken) {
        return false
      }

      const response = await fetch(`${this.baseURL}/api/users/refresh`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          refresh_token: refreshToken
        })
      })

      if (response.ok) {
        const data = await response.json()
        if (data.success && data.access_token) {
          // 更新访问token，保持刷新token不变
          this.saveTokens(data.access_token, refreshToken)
          console.log('✅ Token刷新成功')
          return true
        }
      }

      console.log('❌ Token刷新失败')
      return false
    } catch (error) {
      console.error('Token刷新异常:', error)
      return false
    }
  }

  /**
   * 处理认证错误
   */
  handleAuthError() {
    // 清除用户数据
    localStorage.removeItem('currentUser')

    // 跳转到登录页（如果不在登录页）
    if (window.location.pathname !== '/login') {
      window.location.href = '/login'
    }
  }

  // GET 请求
  async get(endpoint, params = {}) {
    const queryString = new URLSearchParams(params).toString()
    const url = queryString ? `${endpoint}?${queryString}` : endpoint
    return this.request(url, { method: 'GET' })
  }

  // POST 请求
  async post(endpoint, data = {}) {
    return this.request(endpoint, {
      method: 'POST',
      body: JSON.stringify(data)
    })
  }

  // PUT 请求
  async put(endpoint, data = {}) {
    return this.request(endpoint, {
      method: 'PUT',
      body: JSON.stringify(data)
    })
  }

  // DELETE 请求
  async delete(endpoint, data = null) {
    const options = { method: 'DELETE' }
    if (data) {
      options.body = JSON.stringify(data)
    }
    return this.request(endpoint, options)
  }
}

// 创建 API 客户端实例
const apiClient = new ApiClient()

// 导出API客户端实例
export default apiClient

/**
 * 用户管理 API
 */
export const userApi = {
  // 获取所有用户
  async getAll() {
    return apiClient.get('/api/users')
  },

  // 根据ID获取用户
  async getById(userId) {
    return apiClient.get(`/api/users/${userId}`)
  },

  // 根据手机号获取用户
  async getByPhone(phone) {
    return apiClient.get(`/api/users/phone/${phone}`)
  },

  // 创建用户
  async create(userData) {
    return apiClient.post('/api/users', userData)
  },

  // 更新用户
  async update(userId, userData) {
    return apiClient.put(`/api/users/${userId}`, userData)
  },

  // 删除用户
  async delete(userId) {
    return apiClient.delete(`/api/users/${userId}`)
  },

  // 用户登录 - 特殊处理，不进行token刷新
  async login(loginData) {
    const url = `${apiClient.baseURL}/api/users/login`

    try {
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(loginData)
      })

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}))
        throw new Error(errorData.detail || `HTTP ${response.status}: ${response.statusText}`)
      }

      return await response.json()
    } catch (error) {
      console.error('登录请求失败:', error)
      throw error
    }
  },

  // 刷新token
  async refreshToken(refreshToken) {
    return apiClient.post('/api/users/refresh', { refresh_token: refreshToken })
  },

  // 用户登出
  async logout() {
    return apiClient.post('/api/users/logout')
  },

  // 获取当前用户信息
  async getCurrentUser() {
    return apiClient.get('/api/users/me')
  }
}

/**
 * 机构管理 API
 */
export const institutionApi = {
  // 获取所有机构
  async getAll() {
    return apiClient.get('/api/institutions')
  },

  // 根据ID获取机构
  async getById(institutionId) {
    return apiClient.get(`/api/institutions/${institutionId}`)
  },

  // 根据机构编号获取机构
  async getByInstitutionId(instId) {
    return apiClient.get(`/api/institutions/institution-id/${instId}`)
  },

  // 根据负责人ID获取机构
  async getByOwner(ownerId) {
    return apiClient.get(`/api/institutions/owner/${ownerId}`)
  },

  // 创建机构
  async create(institutionData) {
    return apiClient.post('/api/institutions', institutionData)
  },

  // 更新机构
  async update(institutionId, institutionData) {
    return apiClient.put(`/api/institutions/${institutionId}`, institutionData)
  },

  // 删除机构
  async delete(institutionId) {
    return apiClient.delete(`/api/institutions/${institutionId}`)
  },

  // 添加机构图片
  async addImage(institutionId, imageData) {
    return apiClient.post(`/api/institutions/${institutionId}/images`, imageData)
  },

  // 删除机构图片
  async deleteImage(institutionId, imageId) {
    return apiClient.delete(`/api/institutions/${institutionId}/images/${imageId}`)
  },

  // 批量创建机构
  async batchCreate(institutions) {
    return apiClient.post('/api/institutions/batch', { institutions })
  },

  // 批量删除机构
  async batchDelete(institutionIds) {
    return apiClient.post('/api/institutions/batch-delete', { institution_ids: institutionIds })
  }
}

/**
 * 系统配置 API
 */
export const configApi = {
  // 获取所有配置（键值对格式）
  async getAll() {
    return apiClient.get('/api/config')
  },

  // 获取配置列表（详细格式）
  async getList() {
    return apiClient.get('/api/config/list')
  },

  // 根据键名获取配置
  async getByKey(configKey) {
    return apiClient.get(`/api/config/${configKey}`)
  },

  // 设置配置项
  async set(configKey, configValue, description = '') {
    return apiClient.post('/api/config/', {
      config_key: configKey,
      config_value: configValue,
      description
    })
  },

  // 批量更新配置
  async updateMultiple(configs) {
    return apiClient.put('/api/config', configs)
  },

  // 删除配置项
  async delete(configKey) {
    return apiClient.delete(`/api/config/${configKey}`)
  },

  // 重置为默认配置
  async reset() {
    return apiClient.post('/api/config/reset')
  }
}

/**
 * 历史数据 API
 */
export const historyApi = {
  // 获取所有历史记录
  async getAll() {
    return apiClient.get('/api/history')
  },

  // 根据月份获取历史记录
  async getByMonth(month) {
    return apiClient.get(`/api/history/${month}`)
  },

  // 保存月度历史记录
  async save(historyData) {
    try {
      console.log(`💾 开始保存历史数据:`, historyData)
      const result = await apiClient.post('/api/history', historyData)
      console.log(`✅ 历史数据保存成功:`, result)
      return result
    } catch (error) {
      console.error(`❌ 历史数据保存失败:`, error)
      throw error
    }
  },

  // 删除指定月份历史记录
  async deleteByMonth(month) {
    return apiClient.delete(`/api/history/${month}`)
  },

  // 清空所有历史记录
  async clearAll() {
    return apiClient.delete('/api/history')
  },

  // 获取历史统计摘要
  async getSummary() {
    return apiClient.get('/api/history/stats/summary')
  },

  // 清理旧历史数据
  async cleanup(keepMonths = 12) {
    return apiClient.post('/api/history/cleanup', { keep_months: keepMonths })
  },

  // 检查指定月份数据是否存在
  async checkMonthExists(month) {
    try {
      console.log(`🔍 检查月份 ${month} 是否存在...`)
      const response = await apiClient.get(`/api/history/${month}`)
      console.log(`✅ 月份 ${month} 存在`)
      return true
    } catch (error) {
      console.log(`📋 检查月份 ${month} 时出现错误:`, error)

      // 如果是404错误，说明数据不存在，返回false
      if (error.message && error.message.includes('指定月份的历史记录不存在')) {
        console.log(`ℹ️ 月份 ${month} 不存在 (404 - 消息匹配)`)
        return false
      }
      if (error.response && error.response.status === 404) {
        console.log(`ℹ️ 月份 ${month} 不存在 (404 - 状态码)`)
        return false
      }

      // 对于其他错误（包括500），也返回false，不阻止保存流程
      console.warn(`⚠️ 检查月份 ${month} 时出现错误，假设不存在:`, error.message)
      return false
    }
  },


}

/**
 * 数据迁移 API
 */
export const migrationApi = {
  // 执行数据迁移
  async migrate(migrationData) {
    return apiClient.post('/api/migration/migrate', migrationData)
  },

  // 检查迁移状态
  async checkStatus() {
    return apiClient.post('/api/migration/check')
  },

  // 清空数据库
  async clearDatabase(confirm = false) {
    return apiClient.post('/api/migration/clear', { confirm })
  }
}

/**
 * 图片服务API
 */
export const imageApi = {
  // 获取图片URL（新格式）
  getImageUrl(imageId) {
    return `${API_BASE_URL}/api/images/${imageId}`
  },

  // 获取图片信息
  async getImageInfo(imageId) {
    return apiClient.get(`/api/images/${imageId}/info`)
  },

  // 上传图片（新的二进制上传）
  async uploadImage(institutionId, file, options = {}) {
    const formData = new FormData()
    formData.append('file', file)

    // 构建查询参数
    const params = new URLSearchParams({
      institution_id: institutionId
    })

    if (options.quality) {
      params.append('quality', options.quality.toString())
    }
    if (options.maxWidth) {
      params.append('max_width', options.maxWidth.toString())
    }

    // 直接使用request方法，不设置Content-Type让浏览器自动处理FormData
    return apiClient.request(`/api/images/upload?${params.toString()}`, {
      method: 'POST',
      body: formData
      // 不传递headers，让API客户端自动添加认证头
    })
  },

  // 删除图片
  async deleteImage(imageId) {
    return apiClient.delete(`/api/images/${imageId}`)
  },

  // 兼容性方法：将Base64数据转换为新格式
  convertBase64ToImageUrl(base64Data) {
    // 如果已经是新格式的URL，直接返回
    if (base64Data && base64Data.startsWith('/api/images/')) {
      return `${API_BASE_URL}${base64Data}`
    }

    // 如果是Base64格式，暂时返回原数据（迁移期间的兼容处理）
    if (base64Data && base64Data.startsWith('data:image/')) {
      return base64Data
    }

    return null
  }
}

// 删除了不必要的健康检查和连接管理器
