import { defineStore } from 'pinia'
import { ref, computed } from 'vue'
import { userApi } from '@/services/api'
import apiClient from '@/services/api'
import { useDataStore } from './data'

/**
 * 用户认证状态管理
 * 处理用户登录、登出、权限验证等功能
 */
export const useAuthStore = defineStore('auth', () => {
  const currentUser = ref(null)
  const viewAsUser = ref(null)

  /**
   * 计算属性：有效用户（优先使用viewAsUser，否则使用currentUser）
   */
  const effectiveUser = computed(() => viewAsUser.value || currentUser.value)

  /**
   * 计算属性：是否已认证
   */
  const isAuthenticated = computed(() => !!currentUser.value)

  /**
   * 计算属性：是否为管理员
   */
  const isAdmin = computed(() => currentUser.value?.role === 'admin')

  /**
   * 计算属性：当前是否在用户视图模式
   */
  const isViewingAsUser = computed(() => !!viewAsUser.value)

  /**
   * 登录后加载数据
   */
  const loadDataAfterLogin = async () => {
    const dataStore = useDataStore()
    console.log('📊 登录成功，开始加载数据...')
    await dataStore.loadData()
    console.log('✅ 数据加载成功')
  }

  /**
   * 用户登录
   * @param {string} phone - 手机号
   * @param {string} password - 密码
   * @returns {boolean} 登录是否成功
   */
  const login = async (phone, password) => {
    try {
      // 使用 API 进行登录验证
      const response = await userApi.login({ phone, password })

      if (response.success && response.user && response.access_token) {
        // 保存用户信息
        currentUser.value = response.user
        localStorage.setItem('currentUser', JSON.stringify(response.user))

        // 保存tokens到API客户端
        apiClient.saveTokens(response.access_token, response.refresh_token)

        // 验证token是否正确保存
        const savedToken = apiClient.getAccessToken()
        if (!savedToken) {
          throw new Error('Token保存失败')
        }
        console.log('🔑 Token已保存，准备加载数据...')

        // 登录成功后加载数据
        await loadDataAfterLogin()

        console.log('✅ 用户登录成功:', response.user.name)
        return true
      }

      console.log('❌ 登录失败:', response.message)
      return false
    } catch (error) {
      console.error('登录请求失败:', error)
      throw error
    }
  }
  
  /**
   * 用户登出
   */
  const logout = async () => {
    try {
      // 调用后端登出接口
      await userApi.logout()
    } catch (error) {
      console.error('登出请求失败:', error)
    } finally {
      // 清除本地数据
      currentUser.value = null
      localStorage.removeItem('currentUser')

      // 清除用户视图状态
      viewAsUser.value = null
      localStorage.removeItem('viewAsUser')

      // 清除tokens
      apiClient.clearTokens()

      console.log('✅ 用户已登出')
    }
  }
  
  /**
   * 恢复登录状态
   * 从 localStorage 中恢复登录状态并验证token有效性
   */
  const restoreAuth = async () => {
    try {
      // 从 localStorage 中获取用户信息
      const savedUser = localStorage.getItem('currentUser')
      const accessToken = apiClient.getAccessToken()

      if (savedUser && accessToken) {
        // 验证token是否有效
        try {
          const userInfo = await userApi.getCurrentUser()
          if (userInfo && userInfo.id) {
            currentUser.value = userInfo
            // 恢复登录状态后加载数据
            await loadDataAfterLogin()
            // 恢复viewAsUser状态
            restoreViewAsUser()
            console.log('✅ 登录状态已恢复:', userInfo.name)
            return
          }
        } catch (error) {
          console.warn('Token验证失败，尝试刷新:', error)
          // Token可能过期，尝试刷新
          const refreshed = await apiClient.refreshAccessToken()
          if (refreshed) {
            try {
              const userInfo = await userApi.getCurrentUser()
              if (userInfo && userInfo.id) {
                currentUser.value = userInfo
                localStorage.setItem('currentUser', JSON.stringify(userInfo))
                // 恢复登录状态后加载数据
                await loadDataAfterLogin()
                // 恢复viewAsUser状态
                restoreViewAsUser()
                console.log('✅ 登录状态已恢复（刷新后）:', userInfo.name)
                return
              }
            } catch (retryError) {
              console.error('刷新后仍无法获取用户信息:', retryError)
            }
          }
        }
      }

      // 恢复失败，清除数据
      currentUser.value = null
      localStorage.removeItem('currentUser')
      apiClient.clearTokens()
      console.log('🔐 未找到有效登录状态，需要重新登录')

    } catch (error) {
      console.error('恢复登录状态失败:', error)
      currentUser.value = null
    }
  }
  
  /**
   * 更新当前用户信息
   * @param {object} userData - 更新的用户数据
   */
  const updateCurrentUser = (userData) => {
    if (currentUser.value) {
      currentUser.value = { ...currentUser.value, ...userData }
      console.log('✅ 当前用户信息已更新')
    }
  }

  /**
   * 切换到指定用户视图（管理员功能）
   * @param {string} userId - 要切换到的用户ID
   */
  const switchToUser = async (userId) => {
    try {
      const { useDataStore } = await import('./data')
      const dataStore = useDataStore()
      const user = dataStore.getUserById(userId)

      if (user && currentUser.value?.role === 'admin') {
        // 设置viewAsUser，保持currentUser不变
        viewAsUser.value = user
        localStorage.setItem('viewAsUser', JSON.stringify(user))
        console.log('✅ 已切换到用户视图:', user.name)
      }
    } catch (error) {
      console.error('切换用户视图失败:', error)
    }
  }

  /**
   * 从用户视图切换回管理员视图
   */
  const switchBackToAdmin = () => {
    viewAsUser.value = null
    localStorage.removeItem('viewAsUser')
    console.log('✅ 已切换回管理员视图')
  }

  /**
   * 恢复viewAsUser状态
   */
  const restoreViewAsUser = () => {
    try {
      const savedViewAsUser = localStorage.getItem('viewAsUser')
      if (savedViewAsUser) {
        viewAsUser.value = JSON.parse(savedViewAsUser)
        console.log('✅ 已恢复用户视图状态:', viewAsUser.value.name)
      }
    } catch (error) {
      console.error('恢复用户视图状态失败:', error)
      localStorage.removeItem('viewAsUser')
    }
  }

  return {
    currentUser,
    viewAsUser,
    effectiveUser,
    isAuthenticated,
    isAdmin,
    isViewingAsUser,
    login,
    logout,
    restoreAuth,
    updateCurrentUser,
    switchToUser,
    switchBackToAdmin,
    restoreViewAsUser,
    loadDataAfterLogin
  }
}) 