import { defineStore } from 'pinia'
import { ref, computed } from 'vue'
import {
  userApi,
  institutionApi,
  configApi,
  historyApi,
  migrationApi,
  imageApi
} from '@/services/api'
import { useAuthStore } from '@/store/auth'

/**
 * 数据管理store - 重构版本
 * 使用 API 服务替代 localStorage 存储
 * 保持与原版本相同的接口，确保前端组件无需修改
 */
export const useDataStore = defineStore('data', () => {
  // 响应式数据
  const users = ref([])
  const institutions = ref([])
  const systemConfig = ref({})
  
  // 状态管理
  const isLoading = ref(false)
  const isOnline = ref(true)
  const migrationStatus = ref({
    isCompleted: false,
    isInProgress: false,
    error: null
  })

  // 删除了不必要的数据库连接检查

  /**
   * 初始化数据库数据
   */
  const initializeDatabaseData = async () => {
    try {
      console.log('🔄 初始化数据库数据...')

      // 检查数据库是否已有数据
      const status = await migrationApi.checkStatus()

      if (status.total_records === 0) {
        console.log('📊 数据库为空，创建默认配置...')

        // 创建默认系统配置
        await configApi.set('initialized', true)
        await configApi.set('version', '8.8.0')
        await configApi.set('hasDefaultData', false)

        console.log('✅ 默认配置创建完成')
      }

      return true
    } catch (error) {
      console.error('初始化数据库数据失败:', error)
      throw error
    }
  }

  /**
   * 从数据库加载数据
   */
  const loadFromDatabase = async () => {
    try {
      isLoading.value = true
      console.log('🔄 从数据库加载数据...')

      // 获取当前用户信息
      const authStore = useAuthStore()
      const currentUser = authStore.currentUser

      if (!currentUser) {
        throw new Error('用户未登录')
      }

      // 根据用户角色加载不同的数据
      let usersData = []
      let institutionsData = []
      let configData = {}

      if (currentUser.role === 'admin') {
        // 管理员加载所有数据
        const [allUsers, allInstitutions, config] = await Promise.all([
          userApi.getAll(),
          institutionApi.getAll(),
          configApi.getAll()
        ])
        usersData = allUsers
        institutionsData = allInstitutions
        configData = config
      } else {
        // 普通用户只加载自己的机构数据和系统配置
        const [userInstitutions, config] = await Promise.all([
          institutionApi.getByOwner(currentUser.id),
          configApi.getAll()
        ])
        usersData = [currentUser] // 只包含当前用户
        institutionsData = userInstitutions
        configData = config
      }

      // 转换机构数据格式（后端使用下划线，前端使用驼峰命名）
      const convertedInstitutions = institutionsData.map(inst => ({
        id: inst.id,
        institutionId: inst.institution_id,
        name: inst.name,
        ownerId: inst.owner_id,
        images: inst.images || [],
        created_at: inst.created_at,
        updated_at: inst.updated_at
      }))

      // 更新本地状态
      users.value = usersData
      institutions.value = convertedInstitutions
      systemConfig.value = configData
      isOnline.value = true

      console.log(`✅ 数据加载完成: ${users.value.length} 个用户, ${institutions.value.length} 个机构`)

      return true

    } catch (error) {
      console.error('从数据库加载数据失败:', error)
      isOnline.value = false
      throw error
    } finally {
      isLoading.value = false
    }
  }

  /**
   * 初始化空数据状态
   */
  const initializeEmptyData = async () => {
    console.log('🔄 初始化空数据状态...')

    // 为了演示目的，加载一些模拟数据
    users.value = [
      {
        id: 'user_1756285723042',
        phone: '18870041111',
        name: '陈锐屏',
        role: 'user',
        institutions: [],
        created_at: '2025-08-27T09:08:43.757484',
        updated_at: '2025-08-28T06:35:47.681937'
      },
      {
        id: 'user_1756286078085',
        phone: '18870042222',
        name: '张田田',
        role: 'user',
        institutions: [],
        created_at: '2025-08-27T09:08:43.757484',
        updated_at: '2025-08-28T06:35:47.681937'
      },
      {
        id: 'user_1756286102502',
        phone: '18870043333',
        name: '余芳飞',
        role: 'user',
        institutions: [],
        created_at: '2025-08-27T09:08:43.757484',
        updated_at: '2025-08-28T06:35:47.681937'
      }
    ]

    institutions.value = [
      {
        id: 'inst_1756352749368_k9zym68t9',
        institutionId: '73873',
        name: '昆明市五华区爱雅仕口腔诊所',
        ownerId: 'user_1756285723042',
        images: [
          {
            id: 'img_1756300086739_vr9Bqux',
            url: 'data:image/jpeg;base64,/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAYEBQYFBAYGBQYHBwYIChAKCgkJChQODwwQFxQYGBcUFhYaHSUfGhsjHBYWICwgIyYnKSopGR8tMC0oMCUoKSj/2wBDAQcHBwoIChMKChMoGhYaKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCj/wAARCAABAAEDASIAAhEBAxEB/8QAFQABAQAAAAAAAAAAAAAAAAAAAAv/xAAUEAEAAAAAAAAAAAAAAAAAAAAA/8QAFQEBAQAAAAAAAAAAAAAAAAAAAAX/xAAUEQEAAAAAAAAAAAAAAAAAAAAA/9oADAMBAAIRAxEAPwCdABmX/9k=',
            uploadTime: '2025-08-28T08:34:46.739000'
          },
          {
            id: 'img_1756300086739_xr8Cqvw',
            url: 'data:image/jpeg;base64,/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAYEBQYFBAYGBQYHBwYIChAKCgkJChQODwwQFxQYGBcUFhYaHSUfGhsjHBYWICwgIyYnKSopGR8tMC0oMCUoKSj/2wBDAQcHBwoIChMKChMoGhYaKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCj/wAARCAABAAEDASIAAhEBAxEB/8QAFQABAQAAAAAAAAAAAAAAAAAAAAv/xAAUEAEAAAAAAAAAAAAAAAAAAAAA/8QAFQEBAQAAAAAAAAAAAAAAAAAAAAX/xAAUEQEAAAAAAAAAAAAAAAAAAAAA/9oADAMBAAIRAxEAPwCdABmX/9k=',
            uploadTime: '2025-08-28T08:35:12.456000'
          }
        ],
        created_at: '2025-08-28T06:22:44.589423',
        updated_at: '2025-08-28T06:29:03.103923'
      },
      {
        id: 'inst_1756352749368_aygusa4m1',
        institutionId: '73950',
        name: '五华区长青口腔诊所',
        ownerId: 'user_1756285723042',
        images: [
          {
            id: 'img_1756300086740_zr9Dqxy',
            url: 'data:image/jpeg;base64,/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAYEBQYFBAYGBQYHBwYIChAKCgkJChQODwwQFxQYGBcUFhYaHSUfGhsjHBYWICwgIyYnKSopGR8tMC0oMCUoKSj/2wBDAQcHBwoIChMKChMoGhYaKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCj/wAARCAABAAEDASIAAhEBAxEB/8QAFQABAQAAAAAAAAAAAAAAAAAAAAv/xAAUEAEAAAAAAAAAAAAAAAAAAAAA/8QAFQEBAQAAAAAAAAAAAAAAAAAAAAX/xAAUEQEAAAAAAAAAAAAAAAAAAAAA/9oADAMBAAIRAxEAPwCdABmX/9k=',
            uploadTime: '2025-08-28T08:36:22.789000'
          }
        ],
        created_at: '2025-08-28T03:45:49.909788',
        updated_at: '2025-08-28T06:29:03.103923'
      },
      {
        id: 'inst_1756352749368_n231ytwxd',
        institutionId: '001',
        name: '温州奥齿泰口腔门诊部（普通合伙）',
        ownerId: 'user_1756286078085',
        images: [],
        created_at: '2025-08-28T03:45:49.909788',
        updated_at: '2025-08-28T06:29:03.103923'
      },
      {
        id: 'inst_1756352749368_g9mf6mnih',
        institutionId: '002',
        name: '武夷山思美达口腔门诊部',
        ownerId: 'user_1756286078085',
        images: [],
        created_at: '2025-08-28T03:45:49.909788',
        updated_at: '2025-08-28T06:29:03.103923'
      },
      {
        id: 'inst_1756352749368_f7xbgr85j',
        institutionId: '003',
        name: '崇川区海虹口腔门诊部',
        ownerId: 'user_1756286102502',
        images: [],
        created_at: '2025-08-28T03:45:49.909788',
        updated_at: '2025-08-28T06:29:03.103923'
      },
      {
        id: 'inst_1756352749368_8cs5ag7pn',
        institutionId: '004',
        name: '大连西岗悦佳口腔诊所',
        ownerId: 'user_1756286102502',
        images: [],
        created_at: '2025-08-28T03:45:49.909788',
        updated_at: '2025-08-28T06:29:03.103923'
      }
    ]

    systemConfig.value = {
      initialized: true,
      version: '8.8.0',
      hasDefaultData: true
    }

    console.log('✅ 模拟数据状态初始化完成')
  }

  /**
   * 主数据加载函数
   */
  const loadData = async () => {
    try {
      console.log('🚀 开始加载数据...')

      // 直接从数据库加载数据
      await loadFromDatabase()

      // 如果数据库为空，初始化基础数据
      if (users.value.length === 0 && institutions.value.length === 0) {
        await initializeDatabaseData()
        // 重新加载数据
        await loadFromDatabase()
      }

      console.log('✅ 数据加载完成')
      return true

    } catch (error) {
      console.error('❌ 数据加载失败:', error)
      throw error
    }
  }

  // 删除了不必要的兼容性函数

  // ========== 用户管理函数 ==========

  /**
   * 获取所有用户
   */
  const getUsers = () => users.value

  /**
   * 根据ID获取用户
   */
  const getUserById = (id) => users.value.find(u => u.id === id)

  /**
   * 添加用户
   */
  const addUser = async (userData) => {
    try {
      const newUserData = {
        id: `user_${Date.now()}`,
        ...userData,
        institutions: userData.institutions || []
      }
      
      const newUser = await userApi.create(newUserData)
      
      // 更新本地状态
      users.value.push(newUser)
      
      return newUser
    } catch (error) {
      console.error('添加用户失败:', error)
      throw error
    }
  }

  /**
   * 更新用户信息
   */
  const updateUser = async (userId, userData) => {
    try {
      const updatedUser = await userApi.update(userId, userData)
      
      // 更新本地状态
      const index = users.value.findIndex(u => u.id === userId)
      if (index !== -1) {
        users.value[index] = updatedUser
      }
      
      return updatedUser
    } catch (error) {
      console.error('更新用户失败:', error)
      throw error
    }
  }

  /**
   * 删除用户
   */
  const deleteUser = async (userId) => {
    try {
      await userApi.delete(userId)
      
      // 更新本地状态
      const index = users.value.findIndex(u => u.id === userId)
      if (index !== -1) {
        users.value.splice(index, 1)
      }
      
      return true
    } catch (error) {
      console.error('删除用户失败:', error)
      throw error
    }
  }

  // ========== 机构管理函数 ==========

  /**
   * 获取所有机构
   */
  const getInstitutions = () => institutions.value

  /**
   * 根据用户ID获取机构
   */
  const getInstitutionsByUserId = (userId) => {
    return institutions.value.filter(inst => inst.ownerId === userId)
  }

  /**
   * 安全获取机构（兼容性函数）
   */
  const getInstitutionsSafely = () => {
    return institutions.value.filter(inst => inst && inst.id)
  }

  /**
   * 添加机构
   */
  const addInstitution = async (institutionData) => {
    try {
      // 检查机构ID是否提供
      if (!institutionData.institutionId) {
        throw new Error('机构ID不能为空')
      }

      // 检查机构ID是否为数字
      if (!/^\d+$/.test(institutionData.institutionId)) {
        throw new Error('机构ID必须为数字')
      }

      // 检查机构ID是否重复
      if (isInstitutionIdExists(institutionData.institutionId)) {
        throw new Error(`机构ID ${institutionData.institutionId} 已存在`)
      }

      const newInstitutionData = {
        id: `inst_${Date.now()}`,
        institution_id: institutionData.institutionId,
        name: institutionData.name,
        owner_id: institutionData.ownerId
      }

      const newInstitution = await institutionApi.create(newInstitutionData)

      // 转换为前端格式
      const frontendInstitution = {
        id: newInstitution.id,
        institutionId: newInstitution.institution_id,
        name: newInstitution.name,
        ownerId: newInstitution.owner_id,
        images: newInstitution.images || []
      }

      // 更新本地状态
      institutions.value.push(frontendInstitution)

      return frontendInstitution
    } catch (error) {
      console.error('添加机构失败:', error)
      throw error
    }
  }

  /**
   * 更新机构信息
   */
  const updateInstitution = async (institutionId, institutionData) => {
    try {
      const updateData = {
        name: institutionData.name,
        institution_id: institutionData.institutionId,
        owner_id: institutionData.ownerId
      }

      const updatedInstitution = await institutionApi.update(institutionId, updateData)

      // 转换为前端格式并更新本地状态
      const index = institutions.value.findIndex(inst => inst.id === institutionId)
      if (index !== -1) {
        institutions.value[index] = {
          id: updatedInstitution.id,
          institutionId: updatedInstitution.institution_id,
          name: updatedInstitution.name,
          ownerId: updatedInstitution.owner_id,
          images: updatedInstitution.images || []
        }
      }

      return institutions.value[index]
    } catch (error) {
      console.error('更新机构失败:', error)
      throw error
    }
  }

  /**
   * 删除机构
   */
  const deleteInstitution = async (institutionId) => {
    try {
      await institutionApi.delete(institutionId)

      // 更新本地状态
      const index = institutions.value.findIndex(inst => inst.id === institutionId)
      if (index !== -1) {
        institutions.value.splice(index, 1)
      }

      return true
    } catch (error) {
      console.error('删除机构失败:', error)
      throw error
    }
  }

  /**
   * 批量添加机构
   */
  const batchAddInstitutions = async (institutionsData) => {
    try {
      // 准备批量创建的数据
      const institutionsToCreate = institutionsData.map(inst => ({
        id: `inst_${Date.now()}_${Math.random().toString(36).substring(2, 11)}`,
        institution_id: inst.institutionId,
        name: inst.name,
        owner_id: inst.ownerId
      }))

      // 调用批量创建API
      const result = await institutionApi.batchCreate(institutionsToCreate)

      // 重新加载数据以获取最新状态
      await loadFromDatabase()

      return result
    } catch (error) {
      console.error('批量添加机构失败:', error)
      throw error
    }
  }

  /**
   * 批量删除机构
   */
  const batchDeleteInstitutions = async (institutionIds) => {
    try {
      // 调用批量删除API
      const result = await institutionApi.batchDelete(institutionIds)

      // 更新本地状态
      institutionIds.forEach(id => {
        const index = institutions.value.findIndex(inst => inst.id === id)
        if (index !== -1) {
          institutions.value.splice(index, 1)
        }
      })

      return result
    } catch (error) {
      console.error('批量删除机构失败:', error)
      throw error
    }
  }

  /**
   * 为机构添加图片（已弃用，用户端现在直接使用imageApi.uploadImage）
   * 保留此方法以防其他地方调用
   */
  const addImageToInstitution = async (institutionId, imageData) => {
    console.warn('⚠️ addImageToInstitution方法已弃用，请使用imageApi.uploadImage')

    try {
      // 重新从数据库加载数据以确保同步
      await loadFromDatabase()
      console.log('✅ 数据已重新加载，确保界面同步')

      return true
    } catch (error) {
      console.error('加载数据失败:', error)
      throw error
    }
  }

  /**
   * 从机构删除图片（已弃用，用户端现在直接使用imageApi.deleteImage）
   * 保留此方法以防其他地方调用
   */
  const removeImageFromInstitution = async (institutionId, imageId) => {
    console.warn('⚠️ removeImageFromInstitution方法已弃用，请使用imageApi.deleteImage')

    try {
      // 重新从数据库加载数据以确保同步
      await loadFromDatabase()
      console.log('✅ 数据已重新加载，确保界面同步')

      return true
    } catch (error) {
      console.error('加载数据失败:', error)
      throw error
    }
  }

  // ========== 工具函数 ==========

  /**
   * 检查机构ID是否存在
   */
  const isInstitutionIdExists = (institutionId) => {
    return institutions.value.some(inst => inst.institutionId === institutionId)
  }

  /**
   * 生成下一个机构ID
   */
  const generateNextInstitutionId = () => {
    const existingIds = institutions.value
      .map(inst => parseInt(inst.institutionId))
      .filter(id => !isNaN(id))
      .sort((a, b) => a - b)

    let nextId = 1
    for (const id of existingIds) {
      if (id === nextId) {
        nextId++
      } else {
        break
      }
    }

    return nextId.toString().padStart(3, '0')
  }

  // 删除了前端的图片重复检测，这应该在后端处理

  // ========== 计分系统 ==========

  /**
   * 计算用户的交互得分
   * 新算法：0张图片=0分，1张及以上图片=1分（二元计分）
   */
  const calculateInteractionScore = (userId) => {
    const userInstitutions = getInstitutionsByUserId(userId)

    if (userInstitutions.length === 0) return 0

    let totalScore = 0

    for (const institution of userInstitutions) {
      const imageCount = institution.images ? institution.images.length : 0

      // 新的互动得分算法：二元计分
      let institutionScore = 0
      if (imageCount === 0) {
        institutionScore = 0  // 0张图片 = 0分
      } else {
        institutionScore = 1  // 1张及以上图片 = 1分
      }

      totalScore += institutionScore
    }

    return totalScore
  }

  /**
   * 计算用户的绩效得分
   * 新公式：绩效得分 = 互动得分 ÷ 名下的带教机构数 × 10
   */
  const calculatePerformanceScore = (userId) => {
    const userInstitutions = getInstitutionsByUserId(userId)
    const institutionCount = userInstitutions.length

    if (institutionCount === 0) return 0

    const interactionScore = calculateInteractionScore(userId)
    return (interactionScore / institutionCount) * 10
  }

  /**
   * 获取所有用户的得分统计
   */
  const getAllUserScores = computed(() => {
    return users.value
      .filter(user => user.role === 'user')
      .map(user => ({
        ...user,
        institutionCount: getInstitutionsByUserId(user.id).length,
        interactionScore: calculateInteractionScore(user.id),
        performanceScore: calculatePerformanceScore(user.id)
      }))
  })

  // ========== 历史数据管理 ==========

  /**
   * 保存当前月份的统计数据到历史记录
   */
  const saveCurrentMonthStats = async () => {
    try {
      const currentDate = new Date()
      const monthKey = `${currentDate.getFullYear()}-${String(currentDate.getMonth() + 1).padStart(2, '0')}`

      console.log(`📊 保存 ${monthKey} 月份统计数据...`)

      // 获取当前所有用户的统计数据
      const currentStats = users.value
        .filter(user => user.role === 'user')
        .map(user => {
          const userInstitutions = getInstitutionsByUserId(user.id)
          return {
            userId: user.id,
            userName: user.name,
            institutionCount: userInstitutions.length,
            interactionScore: calculateInteractionScore(user.id),
            performanceScore: calculatePerformanceScore(user.id),
            institutions: userInstitutions.map(inst => ({
              id: inst.id,
              name: inst.name,
              imageCount: inst.images ? inst.images.length : 0
            }))
          }
        })

      // 保存完整的机构和图片数据
      const institutionsWithImages = institutions.value.map(inst => ({
        id: inst.id,
        institutionId: inst.institutionId,
        name: inst.name,
        ownerId: inst.ownerId,
        images: inst.images || []
      }))

      const historyData = {
        month: monthKey,
        save_time: new Date().toISOString(),
        total_users: currentStats.length,
        total_institutions: institutions.value.length,
        total_images: institutions.value.reduce((total, inst) => total + (inst.images ? inst.images.length : 0), 0),
        user_stats: currentStats,
        institutions_data: institutionsWithImages, // 新增：保存完整的机构和图片数据
        save_type: 'monthly_reset' // 标识这是月度重置保存
      }

      // 保存到数据库
      await historyApi.save(historyData)

      console.log(`✅ ${monthKey} 月份统计数据保存成功`)
      return true
    } catch (error) {
      console.error('保存历史统计数据失败:', error)
      return false
    }
  }

  /**
   * 获取历史统计数据
   */
  const getHistoryStats = async () => {
    try {
      return await historyApi.getAll()
    } catch (error) {
      console.error('获取历史统计数据失败:', error)
      return []
    }
  }

  /**
   * 获取可用的历史月份
   */
  const getAvailableHistoryMonths = async () => {
    try {
      const historyData = await historyApi.getAll()
      return historyData.map(item => item.month).sort().reverse()
    } catch (error) {
      console.error('获取历史月份失败:', error)
      return []
    }
  }

  /**
   * 获取指定月份的统计数据
   */
  const getMonthStats = async (month) => {
    try {
      const data = await historyApi.getByMonth(month)
      if (data) {
        // 转换字段名从下划线到驼峰命名
        return {
          ...data,
          userStats: data.user_stats || [],
          institutionsData: data.institutions_data || [],  // 转换机构数据字段
          saveTime: data.save_time  // 转换保存时间字段
        }
      }
      return null
    } catch (error) {
      console.error('获取月份统计数据失败:', error)
      return null
    }
  }

  /**
   * 删除指定月份的统计数据
   */
  const deleteMonthStats = async (month) => {
    try {
      await historyApi.deleteByMonth(month)
      return true
    } catch (error) {
      console.error('删除月份统计数据失败:', error)
      return false
    }
  }

  /**
   * 清空所有历史统计数据
   */
  const clearAllHistoryStats = async () => {
    try {
      await historyApi.clearAll()
      return true
    } catch (error) {
      console.error('清空所有历史统计数据失败:', error)
      return false
    }
  }

  /**
   * 手动保存当前月份的统计数据到历史记录
   * @param {boolean} forceOverwrite - 是否强制覆盖已存在的数据
   * @returns {Object} 保存结果 { success: boolean, message: string, isOverwrite: boolean }
   */
  const saveCurrentMonthStatsManually = async (forceOverwrite = false) => {
    try {
      const currentDate = new Date()
      const monthKey = `${currentDate.getFullYear()}-${String(currentDate.getMonth() + 1).padStart(2, '0')}`

      console.log(`📊 手动保存 ${monthKey} 月份统计数据...`)

      // 检查当前月份数据是否已存在
      console.log(`🔍 检查月份 ${monthKey} 数据是否存在...`)
      const monthExists = await historyApi.checkMonthExists(monthKey)
      console.log(`📋 月份 ${monthKey} 存在状态: ${monthExists}`)

      if (monthExists && !forceOverwrite) {
        console.log(`⚠️ 月份 ${monthKey} 数据已存在，需要用户确认覆盖`)
        return {
          success: false,
          message: `${monthKey} 月份的数据已存在，是否要覆盖？`,
          isOverwrite: true,
          monthKey
        }
      }

      // 获取当前所有用户的统计数据
      const currentStats = users.value
        .filter(user => user.role === 'user')
        .map(user => {
          const userInstitutions = getInstitutionsByUserId(user.id)
          return {
            userId: user.id,
            userName: user.name,
            institutionCount: userInstitutions.length,
            interactionScore: calculateInteractionScore(user.id),
            performanceScore: calculatePerformanceScore(user.id),
            institutions: userInstitutions.map(inst => ({
              id: inst.id,
              name: inst.name,
              imageCount: inst.images ? inst.images.length : 0
            }))
          }
        })

      // 保存完整的机构和图片数据
      const institutionsWithImages = institutions.value.map(inst => ({
        id: inst.id,
        institutionId: inst.institutionId,
        name: inst.name,
        ownerId: inst.ownerId,
        images: inst.images || []
      }))

      const historyData = {
        month: monthKey,
        save_time: new Date().toISOString(),
        total_users: currentStats.length,
        total_institutions: institutions.value.length,
        total_images: institutions.value.reduce((total, inst) => total + (inst.images ? inst.images.length : 0), 0),
        user_stats: currentStats,
        institutions_data: institutionsWithImages,
        save_type: 'manual' // 标识这是手动保存
      }

      // 保存到数据库
      console.log(`💾 开始保存 ${monthKey} 月份数据到数据库...`)
      console.log('📊 保存的数据概览:', {
        month: monthKey,
        totalUsers: historyData.total_users,
        totalInstitutions: historyData.total_institutions,
        totalImages: historyData.total_images,
        userStatsCount: historyData.user_stats.length
      })

      await historyApi.save(historyData)

      const message = monthExists ?
        `✅ ${monthKey} 月份统计数据覆盖保存成功` :
        `✅ ${monthKey} 月份统计数据保存成功`

      console.log(message)
      return {
        success: true,
        message,
        isOverwrite: false,
        monthKey
      }
    } catch (error) {
      console.error('手动保存历史统计数据失败:', error)
      return {
        success: false,
        message: `保存失败：${error.message || '未知错误'}`,
        isOverwrite: false
      }
    }
  }

  // ========== 数据管理 ==========

  /**
   * 手动月度重置
   * 保存当前统计数据并清空所有图片上传记录
   */
  const manualMonthlyReset = async () => {
    try {
      console.log('🔄 开始执行月度重置...')

      // 1. 重新加载数据以确保获取最新状态
      console.log('📊 重新加载数据以确保同步...')
      await loadFromDatabase()

      // 2. 保存当前月份的统计数据到历史记录
      const saveResult = await saveCurrentMonthStats()
      if (!saveResult) {
        throw new Error('保存历史统计数据失败')
      }

      // 3. 统计要清除的图片数量
      let clearedImageCount = 0
      let actualDeletedCount = 0
      institutions.value.forEach(inst => {
        if (inst.images && inst.images.length > 0) {
          clearedImageCount += inst.images.length
        }
      })

      console.log(`📊 预计清除 ${clearedImageCount} 张图片`)

      // 4. 批量删除数据库中的所有图片记录（使用新的图片API）
      for (const institution of institutions.value) {
        if (institution.images && institution.images.length > 0) {
          console.log(`🗑️ 正在删除机构 ${institution.name} 的 ${institution.images.length} 张图片...`)

          // 删除该机构的所有图片
          for (const image of institution.images) {
            try {
              console.log(`🗑️ 删除图片: ${image.id}`)
              // 使用新的图片删除API
              await imageApi.deleteImage(image.id)
              actualDeletedCount++
              console.log(`✅ 图片 ${image.id} 删除成功`)
            } catch (error) {
              // 如果图片不存在（404错误），忽略该错误并继续
              if (error.message && error.message.includes('不存在')) {
                console.log(`⚠️ 图片 ${image.id} 已不存在，跳过删除`)
                actualDeletedCount++ // 仍然计入删除数量，因为目标已达成
              } else {
                console.error(`❌ 删除图片 ${image.id} 失败:`, error)
                // 对于其他错误，记录但不中断整个流程
              }
            }
          }
        }
      }

      // 5. 重新加载数据以确保状态同步
      console.log('🔄 重新加载数据以确保状态同步...')
      await loadFromDatabase()

      console.log(`✅ 月度重置完成，实际删除 ${actualDeletedCount} 张图片（预计 ${clearedImageCount} 张）`)

      return {
        success: true,
        clearedCount: actualDeletedCount,
        message: '月度重置成功'
      }

    } catch (error) {
      console.error('❌ 月度重置失败:', error)
      return {
        success: false,
        error: error.message || '月度重置失败',
        clearedCount: 0
      }
    }
  }

  /**
   * 清空所有数据（重置系统）
   */
  const clearAllData = async () => {
    try {
      await migrationApi.clearDatabase(true)

      users.value = []
      institutions.value = []
      systemConfig.value = {}

      console.log('✅ 所有数据已清空')
      return true
    } catch (error) {
      console.error('清空数据失败:', error)
      return false
    }
  }

  /**
   * 重置为默认数据（只保留管理员）
   */
  const resetToDefault = async () => {
    try {
      await clearAllData()
      await initializeDatabaseData()
      console.log('✅ 系统已重置，只保留管理员用户')
      return true
    } catch (error) {
      console.error('重置数据失败:', error)
      return false
    }
  }

  /**
   * 导出数据（用于备份）
   */
  const exportData = () => {
    try {
      const exportData = {
        users: users.value,
        institutions: institutions.value,
        systemConfig: systemConfig.value,
        exportTime: new Date().toISOString(),
        version: '8.8.0'
      }
      return JSON.stringify(exportData, null, 2)
    } catch (error) {
      console.error('导出数据失败:', error)
      return null
    }
  }

  /**
   * 导入数据（用于恢复）
   */
  const importData = async (jsonData) => {
    try {
      console.log('🔄 开始导入数据...')
      const data = JSON.parse(jsonData)

      if (!data.users || !data.institutions || !data.systemConfig) {
        throw new Error('数据格式不正确，缺少必要字段')
      }

      // 准备迁移数据格式
      const migrationData = {
        users: data.users,
        institutions: data.institutions,
        systemConfig: data.systemConfig,
        historyStats: data.historyStats || []
      }

      console.log(`📊 准备导入: ${data.users.length} 个用户, ${data.institutions.length} 个机构`)

      // 使用迁移API将数据写入数据库
      const result = await migrationApi.migrate(migrationData)

      if (result && result.success) {
        console.log('✅ 数据已成功写入数据库')

        // 重新从数据库加载数据以确保同步
        await loadFromDatabase()

        console.log('✅ 数据导入完成')
        return true
      } else {
        throw new Error(result?.message || '数据迁移失败')
      }
    } catch (error) {
      console.error('❌ 导入数据失败:', error)
      return false
    }
  }

  // ========== 兼容性函数 ==========
  // 这些函数保持与原版本相同的接口，确保前端组件无需修改

  // ========== 导出所有函数 ==========
  return {
    // 响应式数据
    users,
    institutions,
    systemConfig,
    isLoading,
    isOnline,
    migrationStatus,

    // 核心函数
    loadData,
    loadFromDatabase,
    initializeDatabaseData,
    initializeEmptyData,

    // 用户管理
    getUsers,
    getUserById,
    addUser,
    updateUser,
    deleteUser,

    // 机构管理
    getInstitutions,
    getInstitutionsByUserId,
    getInstitutionsSafely,
    addInstitution,
    updateInstitution,
    deleteInstitution,
    batchAddInstitutions,
    batchDeleteInstitutions,
    addImageToInstitution,
    removeImageFromInstitution,

    // 工具函数
    isInstitutionIdExists,
    generateNextInstitutionId,

    // 计分系统
    calculateInteractionScore,
    calculatePerformanceScore,
    getAllUserScores,

    // 历史数据
    saveCurrentMonthStats,
    saveCurrentMonthStatsManually,
    getHistoryStats,
    getAvailableHistoryMonths,
    getMonthStats,
    deleteMonthStats,
    clearAllHistoryStats,

    // 数据管理
    clearAllData,
    resetToDefault,
    exportData,
    importData,
    manualMonthlyReset
  }
})
