/**
 * 数据清理工具
 * 用于清理系统中的用户和机构数据，只保留管理员
 */

import { useDataStore } from '@/store/data.js'
import { ElMessage, ElMessageBox } from 'element-plus'

/**
 * 清理所有非管理员数据
 */
export const cleanupAllNonAdminData = async () => {
  try {
    const result = await ElMessageBox.confirm(
      '此操作将删除所有用户数据和机构数据，只保留管理员账户。此操作不可恢复，确定要继续吗？',
      '数据清理确认',
      {
        confirmButtonText: '确定清理',
        cancelButtonText: '取消',
        type: 'warning',
        dangerouslyUseHTMLString: true
      }
    )

    if (result === 'confirm') {
      const dataStore = useDataStore()
      
      // 清理数据
      const success = dataStore.cleanupNonAdminData()
      
      if (success) {
        ElMessage.success('数据清理完成，只保留了管理员账户')
        
        // 清理浏览器缓存
        clearBrowserCache()
        
        // 刷新页面
        setTimeout(() => {
          window.location.reload()
        }, 1500)
        
        return true
      } else {
        ElMessage.error('数据清理失败')
        return false
      }
    }
  } catch (error) {
    if (error !== 'cancel') {
      console.error('数据清理过程中出错:', error)
      ElMessage.error('数据清理过程中出错')
    }
    return false
  }
}

/**
 * 清理浏览器缓存（数据库模式）
 */
export const clearBrowserCache = () => {
  try {
    // 数据库模式下只清理 sessionStorage 和其他临时缓存
    sessionStorage.clear()

    // 清理可能残留的旧版本 localStorage 数据
    const keysToRemove = []
    for (let i = 0; i < localStorage.length; i++) {
      const key = localStorage.key(i)
      if (key && key.startsWith('score_system_')) {
        keysToRemove.push(key)
      }
    }

    keysToRemove.forEach(key => {
      localStorage.removeItem(key)
    })

    console.log('✅ 浏览器缓存已清理（数据库模式）')
  } catch (error) {
    console.error('清理浏览器缓存失败:', error)
  }
}

// 删除了不必要的数据完整性检查

/**
 * 导出当前数据（用于备份）
 */
export const exportCurrentData = () => {
  try {
    const dataStore = useDataStore()
    const exportData = dataStore.exportData()
    
    const blob = new Blob([JSON.stringify(exportData, null, 2)], {
      type: 'application/json'
    })
    
    const url = URL.createObjectURL(blob)
    const a = document.createElement('a')
    a.href = url
    a.download = `score_system_backup_${new Date().toISOString().slice(0, 19).replace(/:/g, '-')}.json`
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    URL.revokeObjectURL(url)
    
    ElMessage.success('数据导出完成')
    return true
  } catch (error) {
    console.error('数据导出失败:', error)
    ElMessage.error('数据导出失败')
    return false
  }
}

/**
 * 重置系统到初始状态
 */
export const resetSystemToInitialState = async () => {
  try {
    const result = await ElMessageBox.confirm(
      '此操作将重置整个系统到初始状态，删除所有数据。此操作不可恢复，确定要继续吗？',
      '系统重置确认',
      {
        confirmButtonText: '确定重置',
        cancelButtonText: '取消',
        type: 'error',
        dangerouslyUseHTMLString: true
      }
    )

    if (result === 'confirm') {
      const dataStore = useDataStore()
      
      // 重置到默认状态
      const success = await dataStore.resetToDefault()

      if (success) {
        ElMessage.success('系统重置完成')

        // 清理浏览器缓存
        clearBrowserCache()

        // 刷新页面
        setTimeout(() => {
          window.location.href = '/'
        }, 1500)

        return true
      } else {
        ElMessage.error('系统重置失败')
        return false
      }
    }
  } catch (error) {
    if (error !== 'cancel') {
      console.error('系统重置过程中出错:', error)
      ElMessage.error('系统重置过程中出错')
    }
    return false
  }
}
