/**
 * 通用工具函数集合
 * 提供各种常用的工具方法
 */

/**
 * 格式化时间
 * @param {string|Date} time - 时间
 * @param {string} format - 格式化模式
 * @returns {string} 格式化后的时间字符串
 */
export const formatTime = (time, format = 'YYYY-MM-DD HH:mm:ss') => {
  const date = new Date(time)
  
  if (isNaN(date.getTime())) {
    return '无效时间'
  }
  
  const year = date.getFullYear()
  const month = String(date.getMonth() + 1).padStart(2, '0')
  const day = String(date.getDate()).padStart(2, '0')
  const hours = String(date.getHours()).padStart(2, '0')
  const minutes = String(date.getMinutes()).padStart(2, '0')
  const seconds = String(date.getSeconds()).padStart(2, '0')
  
  return format
    .replace('YYYY', year)
    .replace('MM', month)
    .replace('DD', day)
    .replace('HH', hours)
    .replace('mm', minutes)
    .replace('ss', seconds)
}

/**
 * 生成唯一ID
 * @param {string} prefix - 前缀
 * @returns {string} 唯一ID
 */
export const generateId = (prefix = 'id') => {
  const timestamp = Date.now()
  const random = Math.random().toString(36).substr(2, 9)
  return `${prefix}_${timestamp}_${random}`
}

/**
 * 防抖函数
 * @param {Function} func - 要防抖的函数
 * @param {number} delay - 延迟时间（毫秒）
 * @returns {Function} 防抖后的函数
 */
export const debounce = (func, delay) => {
  let timeoutId
  return function (...args) {
    clearTimeout(timeoutId)
    timeoutId = setTimeout(() => func.apply(this, args), delay)
  }
}

/**
 * 节流函数
 * @param {Function} func - 要节流的函数
 * @param {number} limit - 时间间隔（毫秒）
 * @returns {Function} 节流后的函数
 */
export const throttle = (func, limit) => {
  let inThrottle
  return function (...args) {
    if (!inThrottle) {
      func.apply(this, args)
      inThrottle = true
      setTimeout(() => inThrottle = false, limit)
    }
  }
}

/**
 * 深拷贝对象
 * @param {any} obj - 要拷贝的对象
 * @returns {any} 拷贝后的对象
 */
export const deepClone = (obj) => {
  if (obj === null || typeof obj !== 'object') {
    return obj
  }
  
  if (obj instanceof Date) {
    return new Date(obj.getTime())
  }
  
  if (obj instanceof Array) {
    return obj.map(item => deepClone(item))
  }
  
  if (typeof obj === 'object') {
    const cloned = {}
    for (const key in obj) {
      if (obj.hasOwnProperty(key)) {
        cloned[key] = deepClone(obj[key])
      }
    }
    return cloned
  }
  
  return obj
}

/**
 * 数据存储工具（已迁移到数据库）
 * 保留接口兼容性，但实际不再使用本地存储
 */
export const storage = {
  /**
   * 获取存储数据（已废弃）
   * @deprecated 已迁移到数据库存储，请使用相应的 API
   */
  get(key, defaultValue = null) {
    console.warn('storage.get 已废弃，请使用数据库 API')
    return defaultValue
  },

  /**
   * 设置存储数据（已废弃）
   * @deprecated 已迁移到数据库存储，请使用相应的 API
   */
  set(key, value) {
    console.warn('storage.set 已废弃，请使用数据库 API')
  },

  /**
   * 删除存储数据（已废弃）
   * @deprecated 已迁移到数据库存储，请使用相应的 API
   */
  remove(key) {
    console.warn('storage.remove 已废弃，请使用数据库 API')
  },

  /**
   * 清空所有存储（已废弃）
   * @deprecated 已迁移到数据库存储，请使用相应的 API
   */
  clear() {
    console.warn('storage.clear 已废弃，请使用数据库 API')
  }
}

/**
 * 文件处理工具
 */
export const fileUtils = {
  /**
   * 读取文件为Base64
   * @param {File} file - 文件对象
   * @returns {Promise<string>} Base64字符串
   */
  readAsDataURL(file) {
    return new Promise((resolve, reject) => {
      const reader = new FileReader()
      reader.onload = () => resolve(reader.result)
      reader.onerror = reject
      reader.readAsDataURL(file)
    })
  },

  /**
   * 验证图片文件
   * @param {File} file - 文件对象
   * @param {number} maxSize - 最大文件大小（MB）
   * @returns {object} 验证结果
   */
  validateImage(file, maxSize = 5) {
    const result = {
      valid: true,
      message: ''
    }

    if (!file.type.startsWith('image/')) {
      result.valid = false
      result.message = '请选择图片文件'
      return result
    }

    if (file.size > maxSize * 1024 * 1024) {
      result.valid = false
      result.message = `图片大小不能超过 ${maxSize}MB`
      return result
    }

    return result
  },

  /**
   * 压缩图片
   * @param {File} file - 图片文件
   * @param {number} quality - 压缩质量 (0-1)
   * @param {number} maxWidth - 最大宽度
   * @returns {Promise<string>} 压缩后的Base64
   */
  compressImage(file, quality = 0.8, maxWidth = 800) {
    return new Promise((resolve) => {
      const canvas = document.createElement('canvas')
      const ctx = canvas.getContext('2d')
      const img = new Image()
      
      img.onload = () => {
        const ratio = Math.min(maxWidth / img.width, maxWidth / img.height)
        canvas.width = img.width * ratio
        canvas.height = img.height * ratio
        
        ctx.drawImage(img, 0, 0, canvas.width, canvas.height)
        resolve(canvas.toDataURL('image/jpeg', quality))
      }
      
      img.src = URL.createObjectURL(file)
    })
  }
}

/**
 * 数据验证工具
 */
export const validator = {
  /**
   * 验证手机号
   * @param {string} phone - 手机号
   * @returns {boolean} 是否有效
   */
  isValidPhone(phone) {
    const phoneRegex = /^1[3-9]\d{9}$/
    return phoneRegex.test(phone)
  },

  /**
   * 验证邮箱
   * @param {string} email - 邮箱地址
   * @returns {boolean} 是否有效
   */
  isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
    return emailRegex.test(email)
  },

  /**
   * 验证密码强度
   * @param {string} password - 密码
   * @returns {object} 验证结果
   */
  validatePassword(password) {
    const result = {
      valid: true,
      message: '',
      strength: 'weak'
    }

    if (password.length < 6) {
      result.valid = false
      result.message = '密码长度不能少于6位'
      return result
    }

    if (password.length >= 8 && /(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/.test(password)) {
      result.strength = 'strong'
    } else if (password.length >= 6 && /(?=.*[a-zA-Z])(?=.*\d)/.test(password)) {
      result.strength = 'medium'
    }

    return result
  }
}

/**
 * 数组工具
 */
export const arrayUtils = {
  /**
   * 数组去重
   * @param {Array} arr - 原数组
   * @param {string} key - 去重依据的键名（对象数组）
   * @returns {Array} 去重后的数组
   */
  unique(arr, key = null) {
    if (!key) {
      return [...new Set(arr)]
    }
    
    const seen = new Set()
    return arr.filter(item => {
      const value = item[key]
      if (seen.has(value)) {
        return false
      }
      seen.add(value)
      return true
    })
  },

  /**
   * 数组排序
   * @param {Array} arr - 原数组
   * @param {string} key - 排序依据的键名
   * @param {string} order - 排序方向 (asc/desc)
   * @returns {Array} 排序后的数组
   */
  sortBy(arr, key, order = 'asc') {
    return [...arr].sort((a, b) => {
      const valueA = a[key]
      const valueB = b[key]
      
      if (order === 'desc') {
        return valueB > valueA ? 1 : valueB < valueA ? -1 : 0
      }
      
      return valueA > valueB ? 1 : valueA < valueB ? -1 : 0
    })
  },

  /**
   * 数组分组
   * @param {Array} arr - 原数组
   * @param {string} key - 分组依据的键名
   * @returns {Object} 分组后的对象
   */
  groupBy(arr, key) {
    return arr.reduce((groups, item) => {
      const group = item[key]
      groups[group] = groups[group] || []
      groups[group].push(item)
      return groups
    }, {})
  }
} 